<?php

/**
 * Form Class for handling custom fields
 *
 * @package        HDP
 * @subpackage     Form
 */
class HDPForm
{
	/**
	 * The array hold list of custom fields
	 *
	 * @var array
	 */
	protected $fields = [];

	/**
	 * Constructor
	 *
	 * @param   array  $fields
	 */
	public function __construct($fields, $config = [])
	{
		foreach ($fields as $field)
		{
			if (empty($field->fieldtype))
			{
				$field->fieldtype = 'Text';
			}

			$class = 'HDPFormField' . ucfirst($field->fieldtype);

			if (class_exists($class))
			{
				$this->fields[$field->name] = new $class($field, $field->default_values);
			}
			else
			{
				throw new RuntimeException('The field type ' . $field->fieldType . ' is not supported');
			}
		}
	}

	/**
	 * Get fields of form
	 *
	 * @return array
	 */
	public function getFields()
	{
		return $this->fields;
	}

	/**
	 * Get the field object from name
	 *
	 * @param   string  $name
	 *
	 * @return HDPFormField
	 */
	public function getField($name)
	{
		return $this->fields[$name];
	}

	/**
	 *
	 * Bind data into form fields
	 *
	 * @param   array  $data
	 * @param   bool   $useDefault
	 *
	 * @return $this
	 */
	public function bind($data, $useDefault = false)
	{
		foreach ($this->fields as $field)
		{
			if (isset($data[$field->name]))
			{
				$field->setValue($data[$field->name]);
			}
			else
			{
				if ($useDefault)
				{
					$field->setValue($field->default_values);
				}
				else
				{
					$field->setValue(null);
				}
			}
		}

		return $this;
	}

	/**
	 * Prepare form fields before being displayed. We need to calculate to see what fields are shown, what fields are hided
	 *
	 * @param   int    $categoryId
	 * @param   array  $relation
	 */
	public function prepareFormField($categoryId = 0, $relation = [])
	{
		foreach ($this->fields as $field)
		{
			$fieldsInCategory = $relation[$categoryId] ?? [];

			if (($field->categoryId != -1) && !in_array($field->id, $fieldsInCategory))
			{
				$field->setVisibility(false);
			}
		}
	}

	/**
	 * Build the custom field dependency
	 *
	 * @return void
	 */
	public function buildFieldsDependency()
	{
		$masterFields = [];
		$fieldsAssoc  = [];
		$hiddenFields = [];

		foreach ($this->fields as $field)
		{
			if (!$field->visible)
			{
				$hiddenFields[] = $field->id;
			}

			if ($field->depend_on_field_id)
			{
				$masterFields[] = $field->depend_on_field_id;
			}

			$fieldsAssoc[$field->id] = $field;
		}

		$masterFields = array_unique($masterFields);

		if (count($masterFields))
		{
			foreach ($this->fields as $field)
			{
				if (in_array($field->id, $masterFields))
				{
					$field->setMasterField(true);

					switch (strtolower($field->type))
					{
						case 'list':
							$field->setAttribute('onchange', "HDP.showHideDependFields($field->id, '$field->name', '$field->type');");
							break;
						case 'radio':
						case 'checkboxes':
							$field->setAttribute('onclick', "HDP.showHideDependFields($field->id, '$field->name', '$field->type');");
							break;
					}
				}

				if ($field->depend_on_field_id && isset($fieldsAssoc[$field->depend_on_field_id]))
				{
					if (in_array($field->depend_on_field_id, $hiddenFields))
					{
						$field->setVisibility(false);
						$hiddenFields[] = $field->id;
					}
					else
					{
						$masterFieldValues = $fieldsAssoc[$field->depend_on_field_id]->value;

						if (is_array($masterFieldValues))
						{
							$selectedOptions = $masterFieldValues;
						}
						elseif (is_string($masterFieldValues) && strpos($masterFieldValues, "\r\n"))
						{
							$selectedOptions = explode("\r\n", $masterFieldValues);
						}
						elseif (is_string($masterFieldValues) && is_array(json_decode($masterFieldValues)))
						{
							$selectedOptions = json_decode($masterFieldValues);
						}
						else
						{
							$selectedOptions = [$masterFieldValues];
						}

						if (is_string($field->depend_on_options) && is_array(json_decode($field->depend_on_options)))
						{
							$dependOnOptions = json_decode($field->depend_on_options);
						}
						else
						{
							$dependOnOptions = explode(',', $field->depend_on_options);
						}

						if (!count(array_intersect($selectedOptions, $dependOnOptions)))
						{
							$field->setVisibility(false);
							$hiddenFields[] = $field->id;
						}
					}
				}
			}
		}
	}

	/**
	 * Method to get form rendered string
	 *
	 * @return string
	 */
	public function render($tableLess = true)
	{
		ob_start();

		foreach ($this->fields as $field)
		{
			echo $field->getControlGroup($tableLess);
		}

		return ob_get_clean();
	}

	/**
	 * Display form fields and it's value
	 *
	 * @param   bool  $tableLess
	 *
	 * @return string
	 */
	public function getOutput($tableLess = true)
	{
		ob_start();
		foreach ($this->fields as $field)
		{
			echo $field->getOutput($tableLess);
		}

		return ob_get_clean();
	}
}