<?php
/**
 * @package        Joomla
 * @subpackage     Helpdesk Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2013 - 2026 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

namespace OSSolution\HelpdeskPro\Site\Helper;

use Joomla\CMS\Factory;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\ParameterType;
use OSL\Utils\Database as DatabaseUtils;

defined('_JEXEC') or die;

class Database
{
	/**
	 * Get all published categories
	 *
	 * @param   string  $order
	 * @param   array   $filters
	 * @param   string  $fieldSuffix
	 * @param   int     $categoryType
	 *
	 * @return array
	 */
	public static function getAllCategories($order = 'title', $filters = [], $fieldSuffix = '', $categoryType = 1)
	{
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('id, parent_id, managers')
			->select($db->quoteName('title' . $fieldSuffix, 'title'))
			->from('#__helpdeskpro_categories')
			->where('published=1')
			->whereIn('category_type', [0, $categoryType])
			->order($order);

		foreach ($filters as $filter)
		{
			$query->where($filter);
		}

		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Get all published statuses
	 *
	 * @param   string  $order
	 * @param   string  $fieldSuffix
	 *
	 * @return array
	 */
	public static function getAllStatuses($order = 'ordering', $fieldSuffix = '')
	{
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('*')
			->from('#__helpdeskpro_statuses')
			->where('published = 1')
			->order($order);

		if ($fieldSuffix)
		{
			DatabaseUtils::getMultilingualFields($query, ['title'], $fieldSuffix);
		}

		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Get all published priorities
	 *
	 * @param   string  $order
	 * @param   string  $fieldSuffix
	 *
	 * @return array
	 */
	public static function getAllPriorities($order = 'ordering', $fieldSuffix = '')
	{
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('id, title')
			->from('#__helpdeskpro_priorities')
			->where('published = 1')
			->order($order);

		if ($fieldSuffix)
		{
			DatabaseUtils::getMultilingualFields($query, ['title'], $fieldSuffix);
		}

		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Get all published labels
	 *
	 * @param   string  $order
	 *
	 * @return array
	 */
	public static function getAllLabels($order = 'title')
	{
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('id, title')
			->from('#__helpdeskpro_labels')
			->where('published=1')
			->order($order);
		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Get all published labels
	 *
	 * @param   int  $staffGroupId
	 *
	 * @return array
	 */
	public static function getAllStaffs($staffGroupId)
	{
		/* @var DatabaseDriver $db */
		$db           = Factory::getContainer()->get('db');
		$config       = Helper::getConfig();
		$displayField = $config->get('staff_display_field', 'username') ?: 'username';

		$query = $db->getQuery(true)
			->select('a.id, a.username, a.name, a.email')
			->from('#__users AS a')
			->innerJoin('#__user_usergroup_map AS b ON a.id = b.user_id')
			->where('group_id=' . (int) $staffGroupId)
			->where('a.block = 0')
			->order($displayField);
		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Get all custom fields which will be displayed in list view
	 *
	 * @param   string  $fieldSuffix
	 *
	 * @return array
	 */
	public static function getFieldsOnListView($fieldSuffix = null)
	{
		$user = Factory::getApplication()->getIdentity();

		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('*')
			->from('#__helpdeskpro_fields')
			->where('show_in_list_view=1')
			->where('published=1')
			->order('ordering');

		if (!$user->authorise('core.admin', 'com_helpdeskpro'))
		{
			$query->whereIn('access', $user->getAuthorisedViewLevels());
		}

		if ($fieldSuffix)
		{
			DatabaseUtils::getMultilingualFields(
				$query,
				['title', 'description', 'values', 'default_values'],
				$fieldSuffix
			);
		}

		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Get name of all searchable custom fields
	 *
	 * @return array
	 */
	public static function getSearchableFields()
	{
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('name')
			->from('#__helpdeskpro_fields')
			->where('published = 1')
			->where('is_searchable = 1');
		$db->setQuery($query);

		return $db->loadColumn();
	}

	/**
	 * Get all filterable custom fields
	 *
	 * @param   string  $fieldSuffix
	 *
	 * @return array
	 */
	public static function getFilterableFields($fieldSuffix = '')
	{
		$user = Factory::getApplication()->getIdentity();
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select($db->quoteName(['id', 'name', 'values', 'fieldtype']))
			->from('#__helpdeskpro_fields')
			->where('published = 1')
			->where('is_filterable = 1')
			->whereIn('fieldtype', Helper::FILTERABLE_FIELD_TYPES, ParameterType::STRING);

		if (!$user->authorise('core.admin', 'com_helpdeskpro'))
		{
			$query->whereIn('access', $user->getAuthorisedViewLevels());
		}

		if ($fieldSuffix)
		{
			DatabaseUtils::getMultilingualFields($query, ['title'], $fieldSuffix);
		}
		else
		{
			$query->select('title');
		}

		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Get a category id by alias
	 *
	 * @param   string  $alias
	 *
	 * @return int
	 */
	public static function getCategoryIdByAlias($alias)
	{
		/* @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get('db');
		$query = $db->getQuery(true)
			->select('id')
			->from('#__helpdeskpro_categories')
			->where('published=1')
			->where('alias = ' . $db->quote($alias));

		$db->setQuery($query);

		return (int) $db->loadResult();
	}
}