<?php
/**
 * @package   OSDownloads-Pro
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2005-2025 Joomlashack.com. All rights reserved
 * @license   https://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of OSDownloads-Pro.
 *
 * OSDownloads-Pro is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * OSDownloads-Pro is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSDownloads-Pro.  If not, see <https://www.gnu.org/licenses/>.
 */

namespace Alledia\OSDownloads\Pro\Joomla\Controller\Admin;

use Alledia\OSDownloads\Factory;
use Alledia\OSDownloads\Free\Joomla\Controller\Admin\Emails as FreeEmailsController;
use Exception;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die();

// phpcs:enable PSR1.Files.SideEffects

class Emails extends FreeEmailsController
{
    public const CSV_BASE_PATH = '/media/com_osdownloads/csv/';

    /**
     * @return void
     * @throws Exception
     */
    public function export()
    {
        $app = Factory::getApplication();
        $db  = Factory::getDatabase();

        $search    = $app->getUserStateFromRequest('osdownloads.email.request.search', 'search', '');
        $cateId    = (int)$app->getUserStateFromRequest('osdownloads.email.request.cateId', 'catid');
        $order     = $app->getUserStateFromRequest('osdownloads.email.order', 'filter_order', 'email.id', 'cmd');
        $orderDir  = $app->getUserStateFromRequest('osdownloads.email.orderDir', 'filter_order_Dir', 'DESC', 'word');
        $confirmed = $app->getUserStateFromRequest('osdownloads.email.confirmed', 'filter_confirmed', '-1', 'int');

        $query = $db->getQuery(true);

        $query->select([
            'email.*',
            'doc.name AS doc_name',
            'cat.id as cat_id',
            'cat.title AS cat_title',
        ]);
        $query->from('#__osdownloads_emails email');
        $query->leftJoin('#__osdownloads_documents doc ON (email.document_id = doc.id)');
        $query->leftJoin('#__categories cat ON (cat.id = doc.catid)');

        if ($search) {
            $search  = $db->quote('%' . $search . '%');
            $orWhere = [
                'email.email LIKE ' . $search,
                'doc.name LIKE ' . $search,
            ];
            $query->where('(%s)', join(' OR ', $orWhere));
        }

        if ($cateId) {
            $query->where('cat.id = ' . $cateId);
        }

        if ($confirmed >= 0) {
            $query->where('email.confirmed = ' . $db->quote($confirmed));
        }

        $query->order($order . ' ' . $orderDir);

        $items = $db->setQuery($query)->loadObjectList();

        $delimiter    = ',';
        $quote        = '"';
        $rowDelimiter = "\n";

        /** @var \OsdownloadsModelEmails $model */
        $model = $this->getModel('Emails');

        $titleRow = [
            'category_id',
            'category_title',
            'confirmed',
            'document_id',
            'document_name',
            'downloaded_on',
            'email',
            'hash',
            'id',
            'synced',
        ];

        $dataRows = [];

        // Add the custom fields names to the title column
        if ($fields = $model->getCustomFieldsNames()) {
            foreach ($fields as $field) {
                $titleRow[] = $field->name;
            }
        }

        foreach ($items as $item) {
            $row = [
                $item->cat_id,
                $item->cat_title,
                $item->confirmed,
                $item->doc_name,
                $item->document_id,
                $item->downloaded_date,
                $item->email,
                $item->hash,
                $item->id,
                $item->synced,
            ];

            $userData = $model->getCustomFieldsValuesByFieldName($item);

            // Check for each field name, so the fields with no value we add a blank column
            foreach ($fields as $field) {
                $row[] = $userData[$field->name] ?? '';
            }

            $dataRows[] = $quote . join($quote . $delimiter . $quote, $row) . $quote;
        }

        // Build the CSV data
        $csv = implode($delimiter, $titleRow) . $rowDelimiter;
        $csv .= implode($rowDelimiter, $dataRows);

        // Determine the file path
        $rand     = bin2hex(openssl_random_pseudo_bytes(64));
        $url      = Route::_('index.php?option=com_osdownloads&view=emails&task=emails.downloadcsv&hash=' . $rand);
        $fileName = JPATH_SITE . static::CSV_BASE_PATH . $rand . '.csv';

        // Remove old files
        if ($files = Folder::files(JPATH_SITE . static::CSV_BASE_PATH, '.csv', false, true)) {
            foreach ($files as $file) {
                File::delete($file);
            }
        }

        File::write($fileName, $csv);

        $message = Text::_('COM_OSDOWNLOADS_LIST_SUCCESSFULLY_EXPORTED')
            . HTMLHelper::_(
                'link',
                $url,
                Text::_('COM_OSDOWNLOADS_DOWNLOAD_CSV_FILE'),
                ['target' => '_blank']
            )
            . Text::_('COM_OSDOWNLOADS_LIST_CSV_WILL_BE_DELETED');

        $this->setRedirect('index.php?option=com_osdownloads&view=emails', $message);
    }

    /**
     * @return void
     * @throws Exception
     */
    public function downloadcsv()
    {
        $app = Factory::getApplication();

        $hash = $app->input->get('hash', '');
        $path = JPATH_SITE . static::CSV_BASE_PATH . '/' . $hash . '.csv';

        if (empty($hash) || !is_file($path)) {
            throw new Exception('Error Processing Download Request');
        }

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=osdownloads_emails.csv');

        echo file_get_contents($path);

        File::delete($path);

        jexit();
    }
}
