<?php
/**
 * 4LOGS
 *
 * @package          4LOGS
 * @copyright        Copyright Weeblr llc - 2021
 * @author           Yannick Gaultier - Weeblr llc
 * @license          GNU General Public License version 3; see LICENSE.md
 * @version          1.1.2.217
 * @date        2021-08-20
 *
 * build 0.0.1
 */

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Language;
use Joomla\CMS\Installer;

// No direct access
defined('_JEXEC') or die;

/**
 * Class Pkg_ForlogsInstallerScript
 *
 * https://docs.joomla.org/J3.x:Developing_an_MVC_Component/Adding_an_install-uninstall-update_script_file
 *
 */
class Pkg_ForlogsInstallerScript
{
	const MIN_PLATFORM_VERSION     = '3.9.0';
	const MAX_PLATFORM_VERSION     = '5.0';
	const INCLUDE_PLATFORM_VERSION = '[]';
	const EXCLUDE_PLATFORM_VERSION = '[]';

	const MIN_PHP_VERSION     = '7.1.0';
	const MAX_PHP_VERSION     = '';
	const INCLUDE_PHP_VERSION = '[]';
	const EXCLUDE_PHP_VERSION = '[]';

	/**
	 * @var string[] Those caches will be cleared at postflight.
	 */
	private $cachesToClean = [
		'4logs_updates'
	];

	/**
	 * Called before any type of action
	 *
	 * @param string    $type
	 * @param \stdClass $parent
	 *
	 * @return bool
	 * @throws Exception
	 */
	public function preflight($type, $parent)
	{
		// check Joomla! version
		if (version_compare(JVERSION, self::MIN_PLATFORM_VERSION, '<') || version_compare(JVERSION, self::MAX_PLATFORM_VERSION, 'ge'))
		{
			Factory::getApplication()
				   ->enqueueMessage(
					   sprintf(
						   '4LOGS requires Joomla! version between %s and %s (you are using %s). Aborting installation',
						   self::MIN_PLATFORM_VERSION, self::MAX_PLATFORM_VERSION, JVERSION
					   ), 'error'
				   );

			return false;
		}

		if (
			version_compare(phpversion(), self::MIN_PHP_VERSION, '<')
			||
			(
				!empty(self::MAX_PHP_VERSION)
				&&
				version_compare(
					phpversion(), self::MAX_PHP_VERSION, 'ge'
				)
			)
		)
		{
			Factory::getApplication()
				   ->enqueueMessage(
					   sprintf(
						   '4LOGS requires PHP version between %s and %s (you are using %s). Aborting installation',
						   self::MIN_PHP_VERSION, self::MAX_PHP_VERSION, phpversion()
					   ), 'error'
				   );

			return false;
		}

		if (function_exists('apc_clear_cache'))
		{
			@apc_clear_cache();
		}

		if (function_exists('opcache_reset'))
		{
			@opcache_reset();
		}

		return true;
	}

	/**
	 * Called after any type of action
	 *
	 * @param string    $type
	 * @param \stdClass $parent
	 *
	 * @return void
	 * @throws Exception
	 */
	public function postflight($type, $parent)
	{
		// auto enable plugins upon first installation
		try
		{
			if ('install' == $type)
			{
				$plugins   = [
					[
						'folder'   => 'system',
						'element'  => 'forlogs',
						'ordering' => 'last'
					]
				];
				$db        = Factory::getDbo();
				$overrides = ['enabled' => 1];

				foreach ($plugins as $plugin)
				{
					$query = $db->getQuery(true);
					$query->select('extension_id')->from('#__extensions')->where($db->qn('type') . '=' . $db->q('plugin'))
						  ->where($db->qn('element') . '=' . $db->q($plugin['element']))
						  ->where($db->qn('folder') . '=' . $db->q($plugin['folder']));
					$db->setQuery($query);
					$pluginId = $db->loadResult();

					if (
						!empty($plugin['ordering'])
						&&
						'last' == $plugin['ordering'])
					{
						$query = $db->getQuery(true);
						$query->select('MAX(ordering)')->from('#__extensions')->where($db->qn('type') . '=' . $db->q('plugin'))
							  ->where($db->qn('folder') . '=' . $db->q($plugin['folder']));
						$db->setQuery($query);
						$currentLastOrdering   = $db->loadResult();
						$overrides['ordering'] = $currentLastOrdering + 1;
					}

					if (!empty($pluginId))
					{
						$extension = Table::getInstance('Extension');
						$extension->load($pluginId);
						$extension->bind($overrides);
						$extension->store();
					}
					else
					{
						Factory::getApplication()->enqueueMessage('4LOGS: Error updating plugin DB record: ' . $plugin['folder'] . ' / ' . $plugin['element']);
					}
				}
			}

			// clear caches, in case cache handling or remote config has been modified
			foreach ($this->cachesToClean as $cacheName)
			{
				Factory::getCache($cacheName)->clean();
			}

			// Build a simple post-install message
			if (in_array($type, ['install', 'discover_install', 'update']))
			{
				$logoUrl         = Uri::root(true) . '/media/plg_system_forlogs/vendor/weeblr/forlogs/assets/images/logo/forlogs.svg';
				$thankYouMessage = Language\Text::_('PLG_SYSTEM_FORLOGS_THANK_YOU_INSTALL');
				$html            = <<<HTML

<h1 style="margin:4rem 0 2rem;text-align:center;">{$thankYouMessage}</h1>
HTML;

				if ('install' == $type)
				{
					$useMsg = Language\Text::sprintf(
						'PLG_SYSTEM_FORLOGS_INSTALL_ADMIN_LINK',
						'index.php?option=com_plugins&view=plugins'
					);
				}
				else
				{
					$useMsg = Language\Text::sprintf(
						'PLG_SYSTEM_FORLOGS_UPDATE_ADMIN_LINK',
						'index.php?option=com_plugins&view=plugins'
					);
				}

				$html .= <<<HTML
<p style="text-align:center;">{$useMsg}</p>
<p style="margin: 4rem 0;text-align: center"><img width="150" src="{$logoUrl}"></p>
HTML;

				echo $html;
			}
		}
		catch (\Exception $e)
		{
			Factory::getApplication()->enqueueMessage('Error: ' . $e->getMessage());
		}
	}

	/**
	 * Called on installation
	 *
	 * @param $parent
	 *
	 * @return void True on success
	 */
	public function install($parent)
	{
		// We run the db update on install in case of re-installing
		// over a past install: db tables already exist but they
		// might be outdated.
		$this->versionSpecificUpdates();
	}

	/**
	 * Called on update
	 *
	 * @param $parent
	 *
	 * @return void True on success
	 */
	public function update($parent)
	{
		$this->versionSpecificUpdates();
	}

	/**
	 * Reconcile db table with current version and runs any version-specific
	 * update code.
	 *
	 * Runs AFTER the main SQL file, DB tables are not created here,
	 * only updated.
	 *
	 * IMPORTANT: Look at this file Git history for actual code performing most common
	 * operation.
	 *
	 */
	private function versionSpecificUpdates()
	{
		$versions = [
			'1.2.2'
		];

		foreach ($versions as $version)
		{
			try
			{
				$methodName = 'update' . str_replace('.', '_', $version);
				$this->{$methodName}();

			}
			catch (\Exception $e)
			{
				Factory::getApplication()->enqueueMessage(
					'Error performing updates, this extension may not work properly. Please try again or contact Weeblr support for assistance. Details: <pre>' . $e->getMessage() . '</pre>'
				);
			}
		}
	}

	/**
	 * Installer plugin removed in version 1.2.2
	 *
	 * @throws Exception
	 */
	private function update1_2_2()
	{
		$this->uninstallPlugin(
			'forlogs',
			'installer'
		);
	}

	/**
	 * Utility to uninstall a plugin.
	 *
	 * @param string $pluginName
	 * @param string $folder
	 * @throws Exception
	 */
	private function uninstallPlugin($pluginName, $folder)
	{
		try
		{
			$db = Factory::getDbo();

			$query = $db->getQuery(true);
			$query->select('*')
				  ->from('#__extensions')
				  ->where($db->qn('type') . '=' . $db->q('plugin'))
				  ->where($db->qn('element') . '=' . $db->q($pluginName))
				  ->where($db->qn('folder') . '=' . $db->q($folder));
			$db->setQuery($query);
			$result = $db->loadAssoc();

			if (empty($result))
			{
				// already removed
				return;
			}

			// remove plugin db id
			$pluginId = $result['extension_id'];

			// dissociate from package or Joomla won't let it be uninstalled.
			$query = $db->getQuery(true);
			$query->update('#__extensions')
				  ->where($db->qn('type') . '=' . $db->q('plugin'))
				  ->where($db->qn('element') . '=' . $db->q($pluginName))
				  ->where($db->qn('folder') . '=' . $db->q($folder))
				  ->set($db->qn('package_id') . ' = 0');
			$db->setQuery($query)->execute();

			// now uninstall
			$installer = new Installer\Installer();
			$installer->uninstall('plugin', $pluginId);
		}
		catch (\Throwable $e)
		{
			Factory::getApplication()->enqueueMessage(
				'Error while removing the 4Logs Installer plugin. It is not needed anymore, so you can probably uninstall it manually using Joomla Manage extensions page. Details: <pre>' . $e->getMessage() . '</pre>',
				'error'
			);
		}
	}
}
