<?php 
/**
 * @package        Joomla
 * @subpackage     Membership Pro
 * @author         Richard Onochie
 * @copyright      Copyright 2018 - 2021 Richard Onochie
 * @license        http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */
defined('_JEXEC') or die('Restricted Access');


class os_coinpayments extends MPFPayment 
{

	/**
	 * Use to store sanitized Coinpayments params
	 *
	 * @var array
	 */
	protected $cp_params = array();


	/**
	 * Constructor
	 *
	 * @param JRegistry $params
	 * @param array     $config
	*/
	public function __construct($params, $config = array()) 
	{

		parent::__construct($params, $config);

		$this->url = 'https://www.coinpayments.net/index.php';

		$this->cp_params['merchant_id'] = trim($params->get('coinpayment_merchant_id', ''));

		$this->cp_params['ipn_secret'] = trim($params->get('coinpayments_ipn_secret', ''));

		$this->cp_params['allowed_currencies'] = trim($params->get('coinpayments_allowed_currencies', ''));

		$this->cp_params['success_url'] = trim($params->get('coinpayments_success_url', ''));

	}




	/**
	 * Process onetime subscription payment
	 *
	 * @param OSMembershipTableSubscriber $row
	 *
	 * @param array   $data
	 */
	public function processPayment($row, $data) {

		$app    = JFactory::getApplication();
		$Itemid = $app->input->getInt('Itemid', 0);

		$siteUrl = JUri::base();
		$ipnUrl = $siteUrl . 'index.php?option=com_osmembership&task=payment_confirm';
		$ipnUrl .= '&payment_method=os_coinpayments';

		$cancelUrl = $siteUrl . 'index.php?option=com_osmembership';
		$cancelUrl .= '&view=cancel&id=' . $row->id . '&Itemid=' . $Itemid;

		$this->setParameter('cmd','_pay_simple');

		$this->setParameter('reset', 1);

		$this->setParameter('merchant', $this->cp_params['merchant_id']);

		$this->setParameter('currency', strtoupper($data['currency']));

		$this->setParameter('amountf', $data['amount']);

		$this->setParameter('item_name', substr($data['item_name'], 0, 127));

		$this->setParameter('custom', $row->id);

		// Make provision for allowed currencies
		if (!empty($this->cp_params['allowed_currencies']))
		{
			$this->setParameter('allow_currencies', $this->cp_params['allowed_currencies']);
		}

		$this->setParameter('want_shipping', 0);

		$this->setParameter('ipn_url', $ipnUrl);

		// Think this through
		if (!empty($this->cp_params['success_url']) && filter_var($this->cp_params['success_url'], FILTER_VALIDATE_URL))
		{
			$this->setParameter('success_url', $this->cp_params['success_url']);
		} 
		else
		{
			$this->setParameter('success_url', $this->getReturnUrl($Itemid));
		}

		$this->setParameter('cancel_url', $cancelUrl);

		// Customer Details
		$this->setParameter('first_name', $row->first_name);

		$this->setParameter('last_name', $row->last_name);

		$this->setParameter('email', $row->email);

		//$this->setParameter('address1', $row->address);
		//$this->setParameter('address2', $row->address2);
		//$this->setParameter('city', $row->city);
		//$this->setParameter('state', $row->state);
		//$this->setParameter('country', $row->country);

		// Send form data
		$this->renderRedirectForm();

	}



	/**
	 * Verify onetime subscription payment
	 *
	 * @return die - IPN callback
	*/
	public function verifyPayment()
	{
		$error_msg = [];
		$ipn_msg = '';

		$app  = JFactory::getApplication();
		$postData = $app->input->post;

		$this->notificationData = $postData->getArray();

		$ipn_mode = $postData->get('ipn_mode', null, 'STRING');

		// Validate IPN
		$response = $this->validateIPN($ipn_mode);
		
		if ($response['status']) 
		{
			$ipn_msg = "IPN OK";

			if ($postData->get('ipn_type', null, 'STRING') != "simple") 
			{
				$error_msg[] = "ipn_type is not equal to simple. ";
			}

			if ($postData->get('merchant', null, 'STRING') != $this->cp_params['merchant_id']) {
				$error_msg[] = "Coinpayments Merchant ID does not match. ";
			}

			/* @var OSMembershipTableSubscriber $row */
			$row  = JTable::getInstance('OsMembership', 'Subscriber');

			// Get the  transaction_id and subscription_id
			$transactionId = $postData->get('txn_id', null, 'STRING');
			$id 				= $postData->get('custom', null, 'INT');

			if (!$row->load($id)) 
			{
				$error_msg[] = "The subscription record cannot be found on our website. ";
			}

			if ($row->published) 
			{
				$error_msg[] = "This subscription has already been activated. ";
			}

			// Make sure each transaction is only processed once
			if ($transactionId && OSMembershipHelper::isTransactionProcessed($transactionId))
			{
				$error_msg[] = "This transaction has already been processed. ";				
			}	

			// check payment_currency
			$expected_cur = strtoupper($row->payment_currency);
			$received_cur = strtoupper($postData->get('currency1', null, 'STRING'));

			if ($received_cur != $expected_cur)
			{
				$error_msg[] = 'Payment currency does not match - saved currency is [' . $expected_cur . '] while payment currency is [' . $received_cur . ']. ';
			}

			// check payment amount
			$expected_amount = $row->gross_amount;
			$received_amount = $postData->get('amount1', null, 'FLOAT');

			if (($expected_amount - $received_amount) > 0.05) 
			{
				$error_msg[] = 'Payment amount does not match - saved amount is [' . $expected_amount . '] while payment amount is [' . $received_amount . ']. ';
			}					

		} 
		else 
		{
			$ipn_msg = 'IPN Error: ' . $response['error_msg'];
			$error_msg[] = $response['error_msg'];
		}
		

		// Give value if everything is okay
		// and kill page since this is IPN callback
		if (empty($error_msg))
		{
			// Check payment status
			$status = $postData->get('status', null, 'INT');

			if ( $status >= 100 || $status == 2 ) 
			{
				$this->onPaymentSuccess($row, $transactionId);
			}

			die($ipn_msg);
		} 
		else 
		{
			if ($this->params->get('ipn_log', 0)) 
			{
				$report = '';

				foreach ($error_msg as $msg) 
				{
					$report .= $msg . "\n";
				}
				
				$report .= "\nTransaction Data Below:\n";

				foreach ($this->notificationData as $key => $value) 
				{
					$report .= $key . ' = ' . html_entity_decode($value, ENT_QUOTES, 'UTF-8'). "\n";
				}

				// Send debug mail
				$config = JFactory::getConfig();
				$from = array( $config->get('mailfrom'), $config->get('fromname'));
				$to = $config->get('mailfrom');
				$subject = $config->get('sitename') . " - MP Coinpayments Plugin Debug Message";
				$mailer = JFactory::getMailer();
				$mailer->setSender($from);
				$mailer->addRecipient($to);
				$mailer->setSubject($subject);
				$mailer->setBody($report);
				$mailer->send();

				// Log error message and post data 
				$this->logGatewayData(implode("\n", $error_msg));

			}

			die($ipn_msg);
			
		}

	}



	/**
	 * Validate post data from gateway
	 *
	 * @param string $ipn_mode
	 *
	 * @return array $resp
	*/
	protected function validateIPN($ipn_mode) {

		$resp = [];
		$resp['status'] = false;
		$resp['error_msg'] = '';

		if ($ipn_mode != 'hmac') 
		{
			$resp['error_msg'] = 'IPN Mode is not HMAC';
			return $resp;
		}

		if (!isset($_SERVER['HTTP_HMAC']) || empty($_SERVER['HTTP_HMAC']))
		{
			$resp['error_msg'] = 'No HMAC signature sent';
			return $resp;
		}

		$request = file_get_contents('php://input');

		if ($request === false || empty($request))
		{
			$resp['error_msg'] = 'Error reading POST data';
			return $resp;
		}

		$hmac = hash_hmac("sha512", $request, $this->cp_params['ipn_secret']);
		// if ($hmac != $_SERVER['HTTP_HMAC']) 
		// Use above if you are running PHP version below 5.6.0 without the hash_equals function
		if (!hash_equals($hmac, $_SERVER['HTTP_HMAC']))
		{
			$resp['error_msg'] = 'HMAC signature does not match';
			return $resp;
		}

		// Everything seems OK
		$resp['status'] = true;
		return $resp;

	}




	/**
	 * Get SEF return URL after processing payment
	 *
	 * @param int $Itemid
	 *
	 * @return string return URL
	 */
	protected function getReturnUrl($Itemid) 
	{

		$rootURL    = rtrim(JUri::root(), '/');
		$subpathURL = JUri::root(true);

		if (!empty($subpathURL) && ($subpathURL != '/'))
		{
			$rootURL = substr($rootURL, 0, -1 * strlen($subpathURL));
		}

		return $rootURL . JRoute::_(OSMembershipHelperRoute::getViewRoute('complete', $Itemid), false);
	}


}