<?php
/**
 * @package     OSL
 * @subpackage  View
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\View;

use Exception;
use Joomla\CMS\Language\Text;
use OSL\Container\Container;
use OSL\Model\Model;

/**
 * Class AbstractView
 *
 * Joomla CMS Base View Class
 */
abstract class AbstractView
{
	/**
	 * Component container
	 *
	 * @var Container
	 */
	protected $container;

	/**
	 * Name of the view
	 *
	 * @var string
	 */
	protected $name;

	/**
	 * The model object.
	 *
	 * @var Model
	 */
	protected $model;

	/**
	 * Determine the view has a model associated with it or not.
	 * If set to No, no model will be created and assigned to the view method when the view is being created
	 *
	 * @var boolean
	 */
	public $hasModel = true;

	/**
	 * Constructor
	 *
	 * @param   Container  $container
	 * @param   array      $config
	 *
	 * @throws Exception
	 */
	public function __construct(Container $container, $config = [])
	{
		$this->container = $container;

		// Set the view name
		if (isset($config['name']))
		{
			$this->name = $config['name'];
		}
		else
		{
			$className = get_class($this);

			$r = null;

			if (!preg_match('/(.*)\\\\View\\\\(.*)\\\\(.*)/i', $className, $r))
			{
				throw new Exception(Text::_('Could not detect the name from class' . $className), 500);
			}

			$this->name = $r[2];
		}

		if (isset($config['has_model']))
		{
			$this->hasModel = $config['has_model'];
		}

		$this->initialize();
	}

	/**
	 * Get name of the current view
	 *
	 * @return string
	 */
	public function getName()
	{
		return $this->name;
	}

	/**
	 * Set the model object
	 *
	 * @param   Model  $model
	 */
	public function setModel(Model $model)
	{
		$this->model = $model;
	}

	/**
	 * Get the model object
	 *
	 * @return Model
	 */
	public function getModel()
	{
		return $this->model;
	}

	/**
	 * Method to escape output.
	 *
	 * @param   string  $output  The output to escape.
	 *
	 * @return string The escaped output.
	 *
	 */
	public function escape($output)
	{
		return $output;
	}

	/**
	 * This blank method give child class a chance to init the class further after being constructed
	 */
	protected function initialize()
	{
	}
}