<?php
/**
 * @package        Joomla
 * @subpackage     Helpdesk Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2013 - 2026 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

namespace OSSolution\HelpdeskPro\Plugin\Finder\HelpdeskPro\Extension;

use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Table\Table;
use Joomla\Component\Finder\Administrator\Indexer\Adapter;
use Joomla\Component\Finder\Administrator\Indexer\Helper;
use Joomla\Component\Finder\Administrator\Indexer\Result;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\DatabaseQuery;
use Joomla\Database\QueryInterface;
use Joomla\Registry\Registry;
use OSSolution\HelpdeskPro\Site\Helper\Route;

defined('_JEXEC') or die;

/**
 * Smart Search adapter plugin for Helpdesk Pro Knowledge Base Articles.
 *
 * @since  6.2.0
 */
class HelpdeskPro extends Adapter
{
	use DatabaseAwareTrait;

	/**
	 * The plugin identifier.
	 *
	 * @var    string
	 * @since  1.0
	 */
	protected $context = 'HelpdeskPro';

	/**
	 * The extension name.
	 *
	 * @var    string
	 * @since  1.0
	 */
	protected $extension = 'com_helpdeskpro';

	/**
	 * The sublayout to use when rendering the results.
	 *
	 * @var    string
	 * @since  1.0
	 */
	protected $layout = 'article';

	/**
	 * The type of content that the adapter indexes.
	 *
	 * @var    string
	 * @since  1.0
	 */
	protected $type_title = 'Knowledge Base Article';

	/**
	 * The table name.
	 *
	 * @var    string
	 * @since  1.0
	 */
	protected $table = '#__helpdeskpro_articles';

	/**
	 * Load the language file on instantiation.
	 *
	 * @var    boolean
	 * @since  1.0
	 */
	protected $autoloadLanguage = true;

	/**
	 * Method to determine if the access level of an item changed.
	 *
	 * @param   string   $context  The context of the content passed to the plugin.
	 * @param   Table    $row      A Table object
	 * @param   boolean  $isNew    If the content has just been created
	 *
	 * @return  void
	 *
	 * @throws  \Exception on database error.
	 */
	public function onFinderAfterSave($context, $row, $isNew): void
	{
		// We only want to handle contacts here
		if ($context === 'com_helpdeskpro.article')
		{
			// Reindex the item
			$this->reindex($row->id);
		}
	}

	/**
	 * Method to setup the indexer to be run.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   1.0
	 */
	protected function setup()
	{
		require JPATH_ADMINISTRATOR . '/components/com_helpdeskpro/bootstrap.php';

		return true;
	}

	/**
	 * Method to index an item. The item must be a Result object.
	 *
	 * @param   Result  $item  The item to index as a Result object.
	 *
	 * @return  void
	 *
	 * @throws  Exception on database error.
	 * @since   1.0
	 */
	protected function index(Result $item)
	{
		static $Itemid;

		if ($Itemid === null)
		{
			$Itemid = Route::findView('articles', 0);
		}

		if (empty($Itemid))
		{
			$Itemid = Route::findView('categories', 0);
		}

		$item->setLanguage();

		// Check if the extension is enabled
		if (ComponentHelper::isEnabled($this->extension) == false)
		{
			return;
		}

		$item->context = 'com_helpdeskpro.article';

		// Initialize the item parameters.
		$item->params = new Registry();

		$item->metadata = new Registry();

		// Build the necessary route and path information.
		$item->url   = $this->getUrl($item->id, $this->extension, $this->layout);
		$item->route = Route::getArticleRoute($item->id, $item->category_id, $Itemid);
		$item->path  = '';

		// Add the type taxonomy data.
		$item->addTaxonomy('Type', 'Knowledge Base Article');

		// Add the category taxonomy data.
		if ($item->category)
		{
			$item->addTaxonomy('Category', $item->category, $item->cat_state, $item->cat_access);
		}

		// Get content extras.
		Helper::getContentExtras($item);

		// Index the item.
		$this->indexer->index($item);
	}

	/**
	 * Method to get the SQL query used to retrieve the list of content items.
	 *
	 * @param   mixed  $query  A DatabaseQuery object or null.
	 *
	 * @return  DatabaseQuery  A database object.
	 *
	 * @since   1.0
	 */
	protected function getListQuery($query = null)
	{
		$db = $this->getDatabase();

		// Check if we can use the supplied SQL query.
		$query = $query instanceof DatabaseQuery ? $query : $db->getQuery(true)
			->select('a.id, a.title, a.alias, a.text AS summary, a.text as body, c.access AS access')
			->select('a.created_date AS start_date, a.category_id')
			->select('a.published AS state, a.hits')
			->select('c.title AS category, c.published AS cat_state, c.access AS cat_access')
			->from('#__helpdeskpro_articles AS a')
			->innerJoin('#__helpdeskpro_categories AS c ON c.id = a.category_id')
			->where('a.published = 1');

		return $query;
	}

	/**
	 * Method to get a SQL query to load the published and access states for
	 * an article and category.
	 *
	 * @return  QueryInterface  A database object.
	 *
	 * @since   1.0
	 */
	protected function getStateQuery()
	{
		$db    = $this->getDatabase();
		$query = $db->getQuery(true);

		// Item ID
		$query->select('a.id');

		// Item and category published state
		$query->select('a.published AS state, c.published AS cat_state');

		// Item and category access levels
		$query->select('1 AS access');
		$query->select('c.access AS cat_access');

		$query->from('#__helpdeskpro_articles AS a');
		$query->leftJoin('#__helpdeskpro_categories AS c ON c.id = a.category_id');

		return $query;
	}

	/**
	 * Method to remove the link information for items that have been deleted.
	 *
	 * @param   string  $context  The context of the action being performed.
	 * @param   object  $table    A Table object.
	 *
	 * @return  boolean  True on success.
	 *
	 * @throws  Exception on database error.
	 * @since   1.0
	 */
	public function onFinderAfterDelete($context, $table)
	{
		if ($context === 'com_helpdeskpro.article')
		{
			$id = $table->id;
		}
		elseif ($context === 'com_finder.index')
		{
			$id = $table->link_id;
		}
		else
		{
			return true;
		}

		// Remove the items.
		return $this->indexer->remove($id);
	}
}
