<?php
/**
 * 4LOGS
 *
 * @package          4LOGS
 * @copyright        Copyright Weeblr llc - 2021
 * @author           Yannick Gaultier - Weeblr llc
 * @license          GNU General Public License version 3; see LICENSE.md
 * @version          1.1.2.217
 * @date        2021-08-20
 */

namespace Weeblr\Forlogs\Api\Controller;

use Weeblr\Forlogs\Model;
use Weeblr\Wblib\V_FORLOGS_217\Api;
use Weeblr\Wblib\V_FORLOGS_217\Wb;
use Weeblr\Wblib\V_FORLOGS_217\System;
use Weeblr\Wblib\V_FORLOGS_217\Fs;

// no direct access
defined('_JEXEC') || defined('WBLIB_EXEC') || die;

class Files extends Api\Controller
{
	/**
	 * Builds up an array of data for use in API response.
	 *
	 * @param Weeblr\Wblib\APi\Request $request
	 * @param array                    $options
	 *
	 * @return array|\Exception
	 */
	public function get($request, $options)
	{
		try
		{
			$path = urldecode(Wb\arrayGet($options, 'path', ''));
			if (empty($path))
			{
				// list all files
				return $this->factory
					->getA(
						Model\Files::class,
						[
							'root' => $this->platform->getLogsPath()
						]
					)->files();
			}
			else
			{
				// get a specific file content
				$download = Wb\arrayGet($options, 'download', false);
				if ('true' === $download)
				{
					$this->downloadFile($path);
				}
				else
				{
					return $this->getFile($path);
				}
			}
		}
		catch (\Throwable $e)
		{
			$this->factory->getThe('forlogs.logger')->error('%s::%d %s - %s', $e->getFile(), $e->getLine(), $e->getMessage(), $e->getTraceAsString());

			return new \Exception('Unable to read the file content. See error log file on server.', System\Http::RETURN_INTERNAL_ERROR);
		}
	}

	/**
	 * Builds up an array of data for use in API response.
	 *
	 * @param Weeblr\Wblib\APi\Request $request
	 * @param array                    $options
	 *
	 * @return array|\Exception
	 */
	public function delete($request, $options)
	{
		try
		{
			$path = urldecode(Wb\arrayGet($options, 'path', ''));
			if (empty($path))
			{
				return new \Exception('No ids provided for delete operation.', System\Http::RETURN_BAD_REQUEST);
			}

			$deleted = $this->factory
				->getA(
					Model\Files::class,
					[
						'path' => $path,
						'root' => $this->platform->getLogsPath()
					]
				)->delete();

			return $deleted instanceof \Throwable
				? $deleted
				: array_merge(
					$deleted,
					['status' => System\Http::RETURN_NO_CONTENT]
				);
		}
		catch (\Throwable $e)
		{
			$this->factory->getThe('forlogs.logger')->error('%s::%d %s - %s', $e->getFile(), $e->getLine(), $e->getMessage(), $e->getTraceAsString());

			return new \Exception('Unable to delete this file. See error log file on server.', System\Http::RETURN_INTERNAL_ERROR);
		}
	}

	protected function getFile($path)
	{
		$content = $this->factory
			->getA(
				Model\File::class,
				[
					'content' => '',
					'path'    => $path,
					'root'    => $this->platform->getLogsPath()
				]
			)->load()
			->content;
		return [
			'data'  => [
				'content' => $content
			],
			'count' => 1,
			'total' => 1,
		];
	}

	protected function downloadFile($path)
	{
		Fs\File::triggerDownload(
			basename($path),
			Wb\slashTrimJoin(
				$this->platform->getLogsPath(),
				$path
			)
		);
	}
}
