<?php
/**
 * 4LOGS
 *
 * @package          4LOGS
 * @copyright        Copyright Weeblr llc - 2021
 * @author           Yannick Gaultier - Weeblr llc
 * @license          GNU General Public License version 3; see LICENSE.md
 * @version          1.1.2.217
 * @date        2021-08-20
 */

namespace Weeblr\Forlogs\Model;


use Weeblr\Wblib\V_FORLOGS_217\Wb;
use Weeblr\Wblib\V_FORLOGS_217\Base;
use Weeblr\Wblib\V_FORLOGS_217\System;
use Weeblr\Wblib\V_FORLOGS_217\Db;

// Security check to ensure this file is being included by a parent file.
defined('_JEXEC') || defined('WBLIB_EXEC') || die;

class Config extends System\Config
{
	/**
	 * @var array Default values for this config, loaded from disk file.
	 */
	protected $defaults = [];

	/**
	 * @var bool Whether to store this config into a keystore for persistence
	 */
	protected $persist = true;

	/**
	 * @var array List of config keys not to be stored (ie hardcoded only)
	 */
	protected $doNotStore = [];

	/**
	 * @var array List of per key type defintion. Optional.
	 */
	protected $enforcedTypes = [];

	/**
	 * Load hardcoded defaults and optionnally read saved values from database.
	 *
	 * @param         $scope
	 * @param bool    $autoload
	 */
	public function __construct($scope, $autoload = true)
	{
		parent::__construct($scope);

		$this->loadDefaults();
	}

	/**
	 * Enforce any type specified for a given key.
	 *
	 * @param array|string $keys
	 * @param mixed        $newValue
	 * @param mixed        $previousValue
	 *
	 * @return mixed
	 */
	protected function enforceTypes($keys, $newValue, $previousValue)
	{
		if (empty($this->enforcedTypes) || !Wb\arrayIsSet($this->enforcedTypes, $keys))
		{
			return;
		}

		switch (Wb\arrayGet($keys, $this->enforcedTypes))
		{
			case System\Convert::STRING:
				$newValue = empty($newValue)
					? ''
					: (is_scalar($newValue)
						? (string)$newValue
						: $newValue
					);
				break;
			case System\Convert::INT:
				$newValue = empty($newValue)
					? ''
					: (is_scalar($newValue)
						? (int)$newValue
						: $newValue
					);
				break;
			case System\Convert::ARRAY:
				$newValue = empty($newValue)
					? []
					: Wb\arrayEnsure($newValue);
				break;
		}

		$this->config = Wb\arraySet(
			$this->config,
			$keys,
			$newValue
		);
	}

	/**
	 * Can trigger an action after a key is set.
	 *
	 * @param array|string $keys
	 * @param mixed        $newValue
	 * @param mixed        $previousValue
	 *
	 * @return Config
	 */
	protected function afterSet($keys, $newValue, $previousValue)
	{
		return $this;
	}

	/**
	 * Validate the configuration content. Used before storing.
	 *
	 * @return $this
	 */
	protected function validate()
	{
		return $this;
	}

	/**
	 * Loads hardcoded default configuration value from /config/{scope}.php.
	 *
	 * @return $this
	 */
	public function loadDefaults()
	{
		$file = FORLOGS_APP_PATH . '/config/' . $this->scope . '.php';
		if (file_exists($file))
		{
			$defaults            = include $file;
			$this->defaults      = Wb\arrayGet($defaults, 'config', []);
			$this->doNotStore    = Wb\arrayGet($defaults, 'doNotStore', []);
			$this->persist       = Wb\arrayGet($defaults, 'persist', true);
			$this->enforcedTypes = Wb\arrayGet($defaults, 'enforcedTypes', []);
			$this->config        = empty($this->defaults) || !is_array($this->defaults) ? $this->config : $this->defaults;
		}

		return $this;
	}

	/**
	 * Encode config current content as json.
	 *
	 * @return false|string
	 */
	public function toJson()
	{
		return json_encode($this->config);
	}

	/**
	 * Returns raw config array.
	 *
	 * @return array
	 */
	public function toArray()
	{
		return $this->config;
	}

	/**
	 * Decode some json and replace/merge with current config.
	 *
	 * @param string $json
	 * @param bool   $merge
	 *
	 * @return $this
	 */
	public function fromJson($json, $merge = true)
	{
		$decoded = json_decode($json, true);
		if (!empty($decoded) && is_array($decoded))
		{
			$this->config = $merge
				? array_merge(
					$this->config,
					$decoded
				)
				: $decoded;
		}

		return $this;
	}
}