<?php

use AcyMailing\Classes\ConfigurationClass;

function acydump($arg, bool $ajax = false, array $options = []): void
{
    $indent = $options['indent'] ?? true;
    $htmlentities = $options['htmlentities'] ?? true;

    ob_start();
    if (is_object($arg) && isset($arg->config)) {
        $safeArg = clone $arg;
        unset($safeArg->config);
        var_dump($safeArg);
    } else {
        var_dump($arg);
    }
    $result = ob_get_clean();

    if ($ajax) {
        $logsPath = ACYM_ROOT.ACYM_LOGS_FOLDER.'debug.log';
        acym_createDir(dirname($logsPath), true, true);

        if (($options['clear_file'] ?? false) === true) {
            file_put_contents($logsPath, '');
        }

        file_put_contents($logsPath, $result, FILE_APPEND);
    } else {
        $style = $indent ? 'margin-left: 220px;' : '';
        echo '<pre style="'.$style.'">';
        echo $htmlentities ? htmlentities($result) : $result;
        echo '</pre>';
    }
}

function acym_debug(bool $file = false, bool $indent = true)
{
    $debug = debug_backtrace();
    $takenPath = [];
    foreach ($debug as $step) {
        if (empty($step['file']) || empty($step['line'])) continue;
        $takenPath[] = $step['file'].' => '.$step['line'];
    }
    acydump(implode("\n", $takenPath), $file, ['indent' => $indent]);
}

function acym_config(bool $reload = false): ConfigurationClass
{
    static $configClass = null;
    if ($configClass === null || $reload) {
        $configClass = new ConfigurationClass();
        $configClass->load();
    }

    return $configClass;
}

function acym_display($messages, string $type = 'success', bool $close = true): void
{
    if (empty($messages)) return;
    if (!is_array($messages)) $messages = [$messages];
    $config = acym_config();
    $remindme = json_decode($config->get('remindme', '[]'), true);
    foreach ($messages as $id => $message) {
        if (strpos($message, 'acym__do__not__remindme') !== false) {
            preg_match('/title="(.*)"/Ui', $message, $matches);
            if (in_array($matches[1], $remindme)) continue;
        }

        echo '<div class="acym__message grid-x acym__message__'.$type.'">';

        if (is_array($message)) $message = implode('</div><div>', $message);

        echo '<div class="cell auto"><div>'.$message.'</div></div>';

        if ($close && strpos($message, 'acym__do__not__remindme') === false) {
            echo '<i data-id="'.acym_escape($id).'" class="cell shrink acym__message__close acymicon-close"></i>';
        }
        echo '</div>';
    }
}

function acym_increasePerf(): int
{
    @ini_set('pcre.backtrack_limit', 1000000);

    $maxExecutionTime = (int)ini_get('max_execution_time');

    if (empty($maxExecutionTime) || $maxExecutionTime < 600) {
        return set_time_limit(600) ? 600 : $maxExecutionTime;
    }

    return $maxExecutionTime;
}

function acym_session(): void
{
    if (empty(session_id()) || session_status() !== PHP_SESSION_ACTIVE) {
        @session_start();
    }
}

function acym_getCID(string $field = ''): int
{
    $oneResult = acym_getVar('array', 'cid', [], '');
    $oneResult = intval(reset($oneResult));
    if (!empty($oneResult) || empty($field)) {
        return $oneResult;
    }

    $oneResult = acym_getVar('int', $field, 0, '');

    return intval($oneResult);
}

function acym_header(string $header, bool $replace = true): void
{
    if (headers_sent()) {
        return;
    }

    header($header, $replace);
}

function acym_getSocialMedias(): array
{
    return json_decode(ACYM_SOCIAL_MEDIA, true);
}

function acym_isAcyCheckerInstalled(): bool
{
    $installed = ACYM_CMS === 'joomla' && acym_isExtensionActive('com_acychecker');

    return $installed || (ACYM_CMS === 'wordpress' && acym_isExtensionActive('acychecker/acychecker.php'));
}

function acym_getErrorLogFilename(string $prefix = ''): string
{
    if (!empty($prefix)) {
        $prefix .= '_';
    }

    return $prefix.'errors.log';
}

function acym_logError(string $message, string $prefix = '', int $maxLines = 0)
{
    $reportPath = acym_getLogPath(acym_getErrorLogFilename($prefix), true);

    if ($maxLines > 0 && file_exists($reportPath)) {
        $lines = file($reportPath);
        if (!empty($lines)) {
            $lines = array_slice($lines, -$maxLines);
            file_put_contents($reportPath, implode("\n", $lines));
        }
    }

    $lr = "\r\n";
    file_put_contents(
        $reportPath,
        $lr.acym_getDate(time()).': '.$message,
        FILE_APPEND
    );
}

function acym_isLogFileErrorExist(string $prefix = ''): bool
{
    $reportPath = acym_getLogPath(acym_getErrorLogFilename($prefix));

    return file_exists($reportPath);
}

function acym_getJsonData(): array
{
    $rawData = file_get_contents('php://input');
    $decodedData = @json_decode($rawData, true);

    return empty($decodedData) ? [] : $decodedData;
}

function displayFreeTrialMessage(): void
{
    if (!acym_isAdmin()) {
        return;
    }

    $config = acym_config();

    if ($config->get('isTrial', 0) != 1) {
        return;
    }

    $expirationDate = $config->get('expirationdate', 0);
    $href = ACYM_ACYMAILING_WEBSITE.'account/license';
    if ($config->get('expirationdate', 0) > time()) {
        $days = ceil(($expirationDate - time()) / 86400);
        $buttonFullAccess = '<a target="_blank" class="button button-secondary margin-left-1 shrink margin-bottom-0" href="'.$href.'">'.acym_translation(
                'ACYM_GET_FULL_ACCESS'
            ).'</a>';
        $message = '<span class="shrink">'.acym_translationSprintf('ACYM_FREE_TRIAL_EXPIRATION_X_DAYS', $days).'</span>';
        $type = 'warning';
    } else {
        $buttonFullAccess = '<a target="_blank" class="button acym__button__upgrade margin-left-1 shrink margin-bottom-0" href="'.$href.'">'.acym_translation(
                'ACYM_GET_FULL_ACCESS'
            ).'</a>';
        $message = '<span class="shrink">'.acym_translation('ACYM_FREE_TRIAL_ENDED').'</span>';
        $type = 'error';
    }
    acym_enqueueMessage('<div class="cell grid-x acym_vcenter">'.$message.$buttonFullAccess.'</div>', $type, false);
}

function acym_removeDashboardNotification(string $name): void
{
    $config = acym_config();
    $existingNotifications = json_decode($config->get('dashboard_notif', '[]'), true) ?? [];

    foreach ($existingNotifications as $key => $existingNotification) {
        if (is_array($existingNotification) && $existingNotification['name'] === $name) {
            unset($existingNotifications[$key]);
        }
    }

    $config->saveConfig(['dashboard_notif' => json_encode($existingNotifications)], false);
}
