<?php
/**
 * Form Field class for the Joomla HDP.
 * Supports a generic list of options.
 *
 * @package     Joomla.HDP
 * @subpackage  Form
 */

use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;

class HDPFormFieldList extends HDPFormField
{

	/**
	 * The form field type.
	 *
	 * @var    string
	 */
	protected $type = 'List';
	/**
	 * This is multiple select?
	 * @var int
	 */
	protected $multiple = 0;
	/**
	 * Options in the form field
	 * @var array
	 */
	protected $options = [];

	/**
	 * Method to instantiate the form field object.
	 *
	 * @param   JTable  $row    the table object store form field definitions
	 * @param   mixed   $value  the initial value of the form field
	 *
	 */
	public function __construct($row, $value)
	{
		parent::__construct($row, $value);

		if ($row->multiple)
		{
			$this->attributes['multiple'] = true;
			$this->multiple               = 1;
		}

		if (is_array($row->values))
		{
			$this->options = $row->values;
		}
		elseif (strpos($row->values, "\r\n") !== false)
		{
			$this->options = explode("\r\n", $row->values);
		}
		else
		{
			$this->options = explode(',', $row->values);
		}

		$this->options = array_map('trim', $this->options);
	}

	/**
	 * Method to get the field input markup.
	 *
	 * @param   HelpdeskProHelperBootstrap  $bootstrapHelper
	 *
	 * @return  string  The field input markup.
	 *
	 */
	protected function getInput($bootstrapHelper = null)
	{
		// Add uk-select if UIKit3 is used
		if ($bootstrapHelper && $bootstrapHelper->getFrameworkClass('uk-select'))
		{
			$this->addClass('uk-select');
		}

		if ($bootstrapHelper && $bootstrapHelper->getFrameworkClass('form-control'))
		{
			$this->addClass('form-control');
		}

		// Get the field options.
		$options    = (array) $this->getOptions();
		$attributes = $this->buildAttributes();

		if ($this->multiple)
		{
			if (is_array($this->value))
			{
				$selectedOptions = $this->value;
			}
			elseif (strpos($this->value, "\r\n"))
			{
				$selectedOptions = explode("\r\n", $this->value);
			}
			elseif (is_string($this->value) && is_array(json_decode($this->value)))
			{
				$selectedOptions = json_decode($this->value);
			}
			else
			{
				$selectedOptions = [$this->value];
			}

			$selectedOptions = array_map('trim', $selectedOptions);
		}
		else
		{
			$selectedOptions = $this->value;
		}

		return HTMLHelper::_(
			'select.genericlist',
			$options,
			$this->name . ($this->multiple ? '[]' : ''),
			trim($attributes),
			'value',
			'text',
			$selectedOptions
		);
	}

	/**
	 * Method to get the field options.
	 *
	 * @return  array  The field option objects.
	 *
	 */
	protected function getOptions()
	{
		$options   = [];
		$options[] = HTMLHelper::_('select.option', '', Text::_($this->row->prompt_text ?: 'HDP_PLEASE_SELECT'));

		foreach ($this->options as $option)
		{
			$options[] = HTMLHelper::_('select.option', trim($option), $option);
		}

		return $options;
	}
}
