<?php
/**
 * @package   OSDownloads-Pro
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2021-2025 Joomlashack.com. All rights reserved
 * @license   https://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of OSDownloads-Pro.
 *
 * OSDownloads-Pro is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * OSDownloads-Pro is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSDownloads-Pro.  If not, see <https://www.gnu.org/licenses/>.
 */

use Alledia\OSDownloads\Factory;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Form\Field\ListField;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Language\Text;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die();

if (class_exists(ListField::class) == false) {
    class_alias(JFormFieldList::class, ListField::class);
}
// phpcs:enable PSR1.Files.SideEffects
// phpcs:disable PSR1.Classes.ClassDeclaration.MissingNamespace

abstract class AbstractFormFieldMailingList extends ListField
{
    /**
     * @var CMSApplication
     */
    protected $app = null;

    /**
     * @var bool
     */
    protected $mailinglistEnabled = null;

    /**
     * @var FormField
     */
    protected $category = null;

    /**
     * @inheritDoc
     * @throws Exception
     */
    public function __construct($form = null)
    {
        $this->app = Factory::getApplication();

        parent::__construct($form);
    }

    /**
     * @inheritDoc
     */
    public function setup(SimpleXMLElement $element, $value, $group = null)
    {
        if (parent::setup($element, $value, $group)) {
            $include = JPATH_ADMINISTRATOR . '/components/com_osdownloads/include.php';
            is_file($include) && (include $include);

            return defined('OSDOWNLOADS_LOADED');
        }

        return false;
    }

    /**
     * @param string $key
     *
     * @return bool
     */
    protected function isMailingListEnabled(string $key): bool
    {
        if ($this->mailinglistEnabled === null) {
            $this->mailinglistEnabled = ($this->form->getName() == 'com_config.component');
            if (!$this->mailinglistEnabled) {
                $this->mailinglistEnabled = (bool)ComponentHelper::getParams('com_osdownloads')
                    ->get("mailinglist.{$key}.enable");
            }
        }

        return $this->mailinglistEnabled;
    }

    /**
     * @param string $key
     * @param array  $options
     *
     * @return void
     */
    protected function checkUseGlobal(string $key, array &$options): void
    {
        if ($this->element['useglobal']) {
            $defaultOption = $this->findOption('', $options);

            if ($defaultOption !== null) {
                if ($this->multiple && \Joomla\CMS\Version::MAJOR_VERSION < 4) {
                    unset($options[$defaultOption]);

                } else {
                    $option = $this->app->input->getCmd('option');

                    if ($option == 'com_osdownloads' && $this->getCategoryField()) {
                        $defaultText = Text::_('COM_OSDOWNLOADS_INHERIT_FROM_CATEGORY');

                    } else {
                        $inheritedIds = (array)ComponentHelper::getParams('com_osdownloads')->get($key);

                        if ($inheritedIds) {
                            $listTexts = [];
                            foreach ($inheritedIds as $inheritedId) {
                                if ($listOption = $this->findOption($inheritedId, $options)) {
                                    $listTexts[] = $options[$listOption]->text;
                                }
                            }

                            if ($listTexts) {
                                $defaultText = Text::sprintf('JGLOBAL_USE_GLOBAL_VALUE', join(', ', $listTexts));
                            }

                        } else {
                            $defaultText = Text::_('COM_OSDOWNLOADS_SELECT_LIST');
                        }
                    }

                    if (empty($defaultText) == false) {
                        $options[$defaultOption]->text = $defaultText;
                    }
                }
            }
        }
    }

    /**
     * @param string   $value
     * @param object[] $options
     *
     * @return ?int
     */
    protected function findOption(string $value, array $options): ?int
    {
        foreach ($options as $index => $option) {
            if ($option->value == $value) {
                return $index;
            }
        }

        return null;
    }

    /**
     * @param ?string $categoryField
     *
     * @return ?FormField
     */
    protected function getCategoryField(?string $categoryField = null): ?FormField
    {
        if ($this->category === null) {
            $this->category = false;

            $categoryField = $categoryField ?: (string)$this->element['categoryField'] ?: 'catid';
            $categoryAtoms = explode('.', $categoryField);

            $fieldName = array_shift($categoryAtoms);
            $groupName = join('.', $categoryAtoms);

            if ($category = $this->form->getField($fieldName, $groupName)) {
                $this->category = $category;
            }
        }

        return $this->category ?: null;
    }
}
