<?php

/**
 * @package   OSDownloads-Pro
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2005-2025 Joomlashack.com. All rights reserved
 * @license   https://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of OSDownloads-Pro.
 *
 * OSDownloads-Pro is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * OSDownloads-Pro is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSDownloads-Pro.  If not, see <https://www.gnu.org/licenses/>.
 */

namespace Alledia\OSDownloads\Pro\Joomla\Controller;

use Alledia\OSDownloads\Factory;
use Alledia\OSDownloads\Free\Joomla\Component\Site as FreeComponentSite;
use Alledia\OSDownloads\Free\Joomla\Controller\Site as FreeControllerSite;
use Alledia\OSDownloads\Free\Joomla\Table\Email;
use Alledia\OSDownloads\Pro\Helper\Helper;
use Exception;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\Registry\Registry;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die();

// phpcs:enable PSR1.Files.SideEffects

class Site extends FreeControllerSite
{
    /**
     * Task called by clicking on the download link sent by e-mail
     *
     * @return void
     * @throws Exception
     */
    public function confirmemail(): void
    {
        $app       = Factory::getApplication();
        $component = FreeComponentSite::getInstance();

        $data = @json_decode(base64_decode($app->input->getBase64('data', '')));

        if ($data) {
            /** @var \OsdownloadsModelItem $modelItem */
            $modelItem = $component->getModel('Item');

            /** @var \OsdownloadsModelEmail $modelEmail */
            $modelEmail = $component->getModel('Email');

            $email = $modelEmail->getEmail((int)$data->email_id);
            $item  = $modelItem->getItem((int)$data->document_id);

            if ($email->id && $item) {
                $email->set('confirmed', $data->hash === $email->get('hash'));
                $email->store();

                $this->notifyAdministrators($item, $email->get('email'), $email);

                $downloadLink = Factory::getPimpleContainer()->helperRoute->getFileDownloadRoute($item->id);
                $app->redirect(Route::_($downloadLink));
            }
        }

        // Invalid data
        $app->input->set('layout', 'error_invalid_data');
        $app->input->set('view', 'item');

        $this->display();
    }

    /**
     * @inheritDoc
     */
    protected function processEmailRequirement(object $item): bool
    {
        $app                  = Factory::getApplication();
        $params               = $app->getParams('com_osdownloads');
        $useEmailConfirmation = (bool)$params->get('use_email_confirmation', 0);
        $component            = FreeComponentSite::getInstance();
        $email                = $app->input->get('require_email', null, 'raw');

        // Must verify the e-mail before download?
        if ($item->require_user_email == 1 || ($item->require_user_email == 2 && !empty($email))) {
            if (!Helper::validateEmail($email)) {
                $app->input->set('layout', 'error_invalid_email');

                return false;
            }

            /** @var \OsdownloadsModelEmail $modelEmail */
            $modelEmail = $component->getModel('Email');
            $emailRow   = $modelEmail->insert($email, $item->id);

            if ($emailRow) {
                // Save the custom fields values.
                PluginHelper::importPlugin('content');
                $app->triggerEvent(
                    'onContentAfterSave',
                    ['com_osdownloads.download', &$emailRow, false, $_POST]
                );

                if ($useEmailConfirmation) {
                    // Send e-mail with a link
                    $mailer = Factory::getMailer();

                    $mailer->addRecipient($email);

                    $data      = [
                        'document_id' => $item->id,
                        'email_id'    => $emailRow->id,
                        'hash'        => $emailRow->hash,
                    ];
                    $linkQuery = [
                        'option' => 'com_osdownloads',
                        'task'   => 'confirmemail',
                        'data'   => base64_encode(json_encode($data)),
                    ];

                    $sefHelper = Factory::getPimpleContainer()->helperSEF;
                    $menu      = $sefHelper->getMenuItemForFile($item->id)
                        ?: $sefHelper->getMenuItemForCategoryTreeRecursively($item->catid);
                    if ($menu) {
                        $linkQuery['Itemid'] = $menu->id;
                    }

                    $downloadLink = Route::_('index.php?' . http_build_query($linkQuery));
                    $downloadUrl  = Uri::getInstance()->toString(['scheme', 'host', 'port']) . $downloadLink;

                    // Subject
                    $emailSubject = $params->get('email_confirmation_subject');
                    if (empty($emailSubject)) {
                        $emailSubject = $app->get('sitename') . ' - ' . Text::_('COM_OSDOWNLOADS_DOWNLOAD_LINK');
                    }

                    // Body
                    $emailBody = $params->get('email_confirmation_body');
                    if (empty($emailBody)) {
                        $emailBody = '<h1>' . $app->get('sitename') . '</h1>';
                        $emailBody .= '<h2>' . $item->name . '</h2>';
                        $emailBody .= Text::_('COM_OSDOWNLOADS_EMAIL_DOWNLOAD_LINK_MESSAGE');
                    }

                    $emailSubject = $this->replaceFileInfoTokens($item, $downloadUrl, $emailSubject);
                    $emailBody    = $this->replaceFileInfoTokens($item, $downloadUrl, $emailBody);


                    if (static::sendEmail($email, $emailSubject, $emailBody)) {
                        $app->input->set('layout', 'thankyou_email_sent');
                    } else {
                        $app->input->set('layout', 'error_sending_email');
                    }

                } else {
                    $emailRow->store();

                    // Download directly
                    $this->notifyAdministrators($item, $email, $emailRow);

                    $app->input->set('layout', 'thankyou');
                }

            } else {
                $app->input->set('layout', 'error_invalid_email');
            }

        } else {
            $this->notifyAdministrators($item, $email);
            $app->input->set('layout', 'thankyou');
        }

        return true;
    }

    /**
     * Replace known shortcode tokens with data from object
     *
     * @param object $item
     * @param string $downloadUrl
     * @param string $text
     *
     * @return string
     */
    protected function replaceFileInfoTokens(object $item, string $downloadUrl, string $text): string
    {
        $replacements = [
            '{{download_url}}' => $downloadUrl,
            '{{name}}'         => empty($item->name) ? '[item Name]' : $item->name,
            '{{category}}'     => empty($item->cat_title) ? '[item Category]' : $item->cat_title,
        ];

        return str_replace(array_keys($replacements), $replacements, $text);
    }

    /**
     * @param array|string $recipients
     * @param string       $subject
     * @param string       $body
     * @param ?bool        $bcc
     *
     * @return bool
     * @throws Exception
     */
    public static function sendEmail($recipients, string $subject, string $body, ?bool $bcc = false): bool
    {
        $app    = Factory::getApplication();
        $params = Factory::getApplication()->getParams('com_osdownloads');
        $mailer = Factory::getMailer();

        $mailer->setSender([
            $params->get('notification.mailfrom') ?: $app->get('mailfrom'),
            $params->get('notification.fromname') ?: $app->get('fromname'),
        ]);

        foreach ((array)$recipients as $recipient) {
            if ($bcc) {
                $mailer->addBcc($recipient);
            } else {
                $mailer->addRecipient($recipient);
            }
        }

        $mailer->isHtml();

        $subject = $subject ?: $app->get('sitename') . ' - ' . Text::_('COM_OSDOWNLOADS_DOWNLOAD_LINK');

        $mailer->setSubject($subject);
        $mailer->setBody($body);

        $result = $mailer->Send();
        if ($result instanceof Exception) {
            Factory::getApplication()->input->set('errorMessage', $result->getMessage());
            return false;
        }

        return $result;
    }

    /**
     * Send administrator notifications according to configuration
     *
     * @param object  $item
     * @param ?string $email
     * @param ?Email  $collectedEmailEntryItem
     *
     * @return void
     * @throws Exception
     */
    protected function notifyAdministrators(
        object $item,
        ?string $email = null,
        ?Email $collectedEmailEntryItem = null
    ): void {
        /** @var Registry $params */
        $params = Factory::getApplication()->getParams('com_osdownloads');
        if ($params->get('notification.enabled')) {
            $db = Factory::getDatabase();

            $recipients = [];
            if ($params->get('notification.system')) {
                // Gather system users
                $recipients = $db->setQuery(
                    $db->getQuery(true)
                        ->select('email')
                        ->from('#__users')
                        ->where([
                            $db->quoteName('sendEmail') . '= 1',
                            $db->quoteName('block') . ' = 0',
                        ])
                )
                    ->loadColumn();
            }

            if ($additional = $params->get('notification.additional')) {
                $recipients = array_unique(
                    array_merge(
                        $recipients,
                        array_filter(
                            array_map('trim', preg_split('/[\r\n,]/', $additional))
                        )
                    )
                );
            }

            if ($recipients) {
                $app  = Factory::getApplication();
                $user = Factory::getUser();

                $body = Text::sprintf(
                    'COM_OSDOWNLOADS_NOTIFICATION_BODY',
                    $item->name,
                    $item->id,
                    $email ?: $user->email ?: Text::_('COM_OSDOWNLOADS_GUEST')
                );

                if ($collectedEmailEntryItem) {
                    $component = FreeComponentSite::getInstance();

                    /** @var \OsdownloadsModelEmails $modelEmails */
                    $modelEmails        = $component->getModel('Emails');
                    $customFieldsValues = $modelEmails->getCustomFieldsValuesAsString($collectedEmailEntryItem);

                    $body .= '<br>' . $customFieldsValues;
                }

                static::sendEmail(
                    $recipients,
                    Text::sprintf('COM_OSDOWNLOADS_NOTIFICATION_SUBJECT', $app->get('sitename')),
                    $body
                );
            }
        }
    }
}
