<?php
/**
 * @package   OSDownloads-Pro
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2005-2025 Joomlashack.com. All rights reserved
 * @license   https://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of OSDownloads-Pro.
 *
 * OSDownloads-Pro is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * OSDownloads-Pro is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSDownloads-Pro.  If not, see <https://www.gnu.org/licenses/>.
 */

namespace Alledia\OSDownloads\Pro\Joomla\Plugin;

use Alledia\OSDownloads\Factory;
use Alledia\OSDownloads\Free as FreeOsdownloads;
use Alledia\OSDownloads\Pro as ProOsdownloads;
use Exception;
use JEventDispatcher;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\Event\DispatcherInterface;
use Joomla\Registry\Registry;
use Throwable;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die();

// phpcs:enable PSR1.Files.SideEffects

class Content extends FreeOsdownloads\Joomla\Plugin\Content
{
    /**
     * @var bool
     */
    protected $supportLoaded = false;

    /**
     * @var bool
     */
    protected $tagsFound = false;

    /**
     * @var FreeOsdownloads\Joomla\Component\Site
     */
    protected $component = null;

    /**
     * PlgContentOSDownloads constructor.
     *
     * @param JEventDispatcher|DispatcherInterface $subject
     * @param array                                $config
     *
     * @throws Exception
     */
    public function __construct($subject, array $config)
    {
        parent::__construct($subject, $config);

        if (
            $this->isEnabled()
            && Factory::getUser()->guest
        ) {
            // Always load support files if under aggressive caching
            $cache = Factory::getCache();
            if (isset($cache->options['caching']) && $cache->options['caching']) {
                $this->loadSupport();
            }
        }
    }

    /**
     * @param string   $context
     * @param object   $item
     * @param Registry $params
     *
     * @return void
     */
    public function onContentBeforeDisplay($context, $item, $params): void
    {
        if (
            $this->isEnabled()
            && isset($item->text)
            && $this->checkTagsFound($item->text)
        ) {
            try {
                $this->loadSupport();
                $this->replaceTags($item->text);

            } catch (Throwable $e) {
                $this->app->enqueueMessage(
                    Text::sprintf('COM_OSDOWNLOADS_ERROR_PLUGIN', __CLASS__, $e->getMessage()),
                    'error'
                );
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function onContentPrepareForm($form, $data): bool
    {
        // Update the form with pro features before other plugins get a chance
        if ($this->app->isClient('administrator')) {
            $formName = $form->getName();

            if (stripos($formName, 'com_osdownloads') === 0) {
                switch ($formName) {
                    case 'com_osdownloads.file':
                        $proForm = JPATH_ADMINISTRATOR . '/components/com_osdownloads/models/forms/file_pro.xml';
                        if (is_file($proForm)) {
                            $form->loadFile($proForm);
                        }
                        break;

                    default:
                        $parts   = explode('.', $formName);
                        $context = array_shift($parts);

                        if ($context == 'com_osdownloads' && $parts) {
                            $parts   = array_reverse($parts);
                            $proForm = sprintf(
                                JPATH_ADMINISTRATOR . '/components/com_osdownloads/models/forms/%s_pro.xml',
                                join('_', $parts)
                            );
                            if (is_file($proForm)) {
                                $form->loadFile($proForm);
                            }
                        }
                }
            }
        }

        return parent::onContentPrepareForm($form, $data);
    }

    /**
     * @param string   $context
     * @param object   $item
     * @param Registry $params
     *
     * @return void
     * @throws Exception
     */
    public function onContentPrepare($context, $item, $params): void
    {
        $this->onContentBeforeDisplay($context, $item, $params);
    }

    /**
     * Load all support css/js required to handle the popup
     * NB: If needed, this must be called no later than onBeforeRender()
     *
     * @return void
     * @throws Exception
     */
    protected function loadSupport()
    {
        if (!$this->supportLoaded) {
            $this->supportLoaded = true;

            // Load jQuery?
            if (Factory::getApplication()->get('jquery') !== true) {
                Factory::getApplication()->set('jquery', true);
                HTMLHelper::_('jquery.framework');
            }

            HTMLHelper::_('behavior.formvalidator');

            $options = [
                'version'  => $this->getComponent()->getMediaVersion(),
                'relative' => true,
            ];

            HTMLHelper::_('script', 'com_osdownloads/jquery.osdownloads.bundle.min.js', $options);
            HTMLHelper::_('stylesheet', 'com_osdownloads/frontend.css', $options);
        }
    }

    /**
     * @param string $text
     *
     * @return bool
     */
    protected function checkTagsFound(string $text): bool
    {
        if ($this->enabled) {
            $this->tagsFound = $this->tagsFound
                || ($text && mb_strpos($text, '{osdownloads ') !== false);

            return $this->tagsFound;
        }

        return false;
    }

    /**
     * Look for recognized shortcodes in the text and replace.
     * If you have more than one download button for the same file, it is
     * advised to have different tags in the content for each one. The
     * easier way is to force a custom class for each button. So both won't
     * be replaced at the same time, keeping different unique ID for each.
     *
     * @param string $text
     *
     * @return bool
     * @throws Exception
     */
    protected function replaceTags(string &$text): bool
    {
        if ($this->checkTagsFound($text) == false) {
            return false;
        }

        $model = $this->getModel();

        // {osdownloads download_button[.css_class.css_class2] 3 "link text"}
        $tagRegex = [
            '#{osdownloads[\s]+download_button([^}\s\n\t]+)?\s([0-9a-z\-_]+)(?:[\s\t]+"([^"]+)")?}#i',
        ];

        $tagsReplaced = false;
        foreach ($tagRegex as $regex) {
            if (preg_match_all($regex, $text, $matches)) {
                $tagsReplaced = true;
                foreach ($matches[0] as $k => $source) {
                    $buttonClasses = trim(implode(' ', explode('.', $matches[1][$k])));
                    $fileId        = (int)$matches[2][$k];
                    $buttonLabel   = preg_replace('/[\'"]/', '', $matches[3][$k]);

                    if (empty($fileId)) {
                        continue;
                    }

                    $file = $model->getItem($fileId);

                    if (empty($file)) {
                        $text = str_replace(
                            $source,
                            Text::sprintf('COM_OSDOWNLOADS_ERROR_FILE_NOT_FOUND', $fileId, 'getFileFromId'),
                            $text
                        );

                    } else {
                        /*
                         * We need to allow each instance of the shortcode
                         * to generate its own unique html due to unique ids
                         * required for each instance.
                         */
                        $data                = new FreeOsdownloads\DisplayData(new Registry());
                        $data->item          = $file;
                        $data->buttonClasses = $buttonClasses;

                        $data->item->download_text = $buttonLabel;

                        $sourceLength = strlen($source);
                        while (($pos = strpos($text, $source)) !== false) {
                            $replacement = sprintf(
                                '<div class="osdownloadsaction"><div class="btn_download">%s</div></div>',
                                LayoutHelper::render(
                                    'osdownloads.buttons.download',
                                    $data,
                                    null,
                                    ['component' => 'com_osdownloads']
                                )
                            );

                            $text = substr_replace($text, $replacement, $pos, $sourceLength);
                        }
                    }
                }
            }
        }

        return $tagsReplaced;
    }

    /**
     * @return FreeOsdownloads\Joomla\Component\Site|Object
     */
    protected function getComponent()
    {
        if ($this->component === null) {
            $this->component = FreeOsdownloads\Joomla\Component\Site::getInstance();
            $this->component->loadLibrary();
        }

        return $this->component;
    }

    /**
     * @param string $name
     *
     * @return ProOsdownloads\Joomla\Model\Item
     */
    protected function getModel(string $name = 'Item')
    {
        return $this->getComponent()->getModel($name);
    }
}
