<?php
/**
 * @package     OSL
 * @subpackage  Controller
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\Controller;

use Exception;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Document\Document;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Session\Session;
use OSL\Container\Container;
use OSL\Input\Input;
use OSL\Model\Model;
use OSL\View\HtmlView;
use ReflectionClass;
use ReflectionMethod;

defined('_JEXEC') or die;

/**
 * Class Controller
 *
 * Base class for a Joomla Controller. Controller (Controllers are where you put all the actual code.) Provides basic
 * functionality, such as rendering views (aka displaying templates).
 */
class Controller
{
	/**
	 * Component container
	 *
	 * @var Container
	 */

	protected $container;
	/**
	 * Array which hold all the controller objects has been created
	 *
	 * @var array
	 */
	protected static $instances = [];

	/**
	 * Name of the controller
	 *
	 * @var string
	 */
	protected $name;

	/**
	 * The application object.
	 *
	 * @var CMSApplication
	 */
	protected $app;

	/**
	 * Controller input
	 *
	 * @var Input
	 */
	protected $input;

	/**
	 * Array of class methods
	 *
	 * @var array
	 */
	protected $methods = [];

	/**
	 * Array which map a task with the method will be called
	 *
	 * @var array
	 */
	protected $taskMap = [];

	/**
	 * Current or most recently performed task.
	 *
	 * @var string
	 */
	protected $task;

	/**
	 * URL for redirection.
	 *
	 * @var string
	 */
	protected $redirect;

	/**
	 * Redirect message.
	 *
	 * @var string
	 */
	protected $message = null;

	/**
	 * Redirect message type.
	 *
	 * @var string
	 */
	protected $messageType = 'message';

	/**
	 * Method to get instance of a controller
	 *
	 * @param   Container  $container
	 * @param   array      $config
	 *
	 * @return Controller
	 *
	 * @throws Exception
	 */
	public static function getInstance(Container $container, array $config = [])
	{
		$input = $container->input;

		// Handle task format controller.task
		$task = $input->get('task', '');
		$pos  = strpos($task, '.');

		if ($pos !== false)
		{
			$name = substr($task, 0, $pos);
			$task = substr($task, $pos + 1);
			$input->set('task', $task);
		}
		else
		{
			$name = $container->inflector->singularize($input->getCmd('view'));

			if (empty($name))
			{
				$name = 'controller';
			}
		}

		$option = $container->option;

		if (!isset(self::$instances[$option . $name]))
		{
			$config['name'] = $name;

			if ($task == '')
			{
				// Always use default controller for display task

				if (class_exists($container->namespace . '\\Override\\Controller\\Controller'))
				{
					$className = $container->namespace . '\\Override\\Controller\\Controller';
				}
				else
				{
					$className = $container->namespace . '\\Controller\\Controller';
				}

				self::$instances[$option . $name] = new $className($container, $config);
			}
			else
			{
				self::$instances[$option . $name] = $container->factory->createController($name, $config);
			}
		}

		return self::$instances[$option . $name];
	}

	/**
	 * Constructor.
	 *
	 * @param   Container  $container
	 *
	 * @param   array      $config  An optional associative array of configuration settings.
	 *
	 * @throws Exception
	 */
	public function __construct(Container $container, array $config = [])
	{
		$this->container = $container;
		$this->app       = Factory::getApplication();
		$input           = $container->input;

		// Build the default taskMap based on the class methods
		$xMethods = get_class_methods('\\OSL\\Controller\Controller');
		$r        = new ReflectionClass($this);

		$rMethods = $r->getMethods(ReflectionMethod::IS_PUBLIC);

		foreach ($rMethods as $rMethod)
		{
			$mName = $rMethod->getName();

			if (!in_array($mName, $xMethods) || $mName == 'display')
			{
				$this->taskMap[strtolower($mName)] = $mName;
				$this->methods[]                   = strtolower($mName);
			}
		}

		// Register controller default task
		if (isset($config['default_task']))
		{
			$this->registerTask('__default', $config['default_task']);
		}
		else
		{
			$this->registerTask('__default', 'display');
		}

		$this->name  = $config['name'];
		$this->input = $input;
		$this->task  = $input->getCmd('task', 'display');
	}

	/**
	 * Execute the given task
	 *
	 * @return $this return itself to support changing
	 *
	 * @throws Exception
	 */
	public function execute()
	{
		$task = strtolower($this->task);

		if (empty($task))
		{
			$task = 'display';
		}

		if (isset($this->taskMap[$task]))
		{
			$this->task = $task;

			$doTask = $this->taskMap[$task];

			$this->$doTask();

			return $this;
		}

		throw new Exception(Text::sprintf('JLIB_APPLICATION_ERROR_TASK_NOT_FOUND', $task), 404);
	}

	/**
	 * Method to display a view
	 *
	 * This function is provide as a default implementation, in most cases
	 * you will need to override it in your own controllers.
	 *
	 * @param   boolean  $cachable   If true, the view output will be cached
	 *
	 * @param   array    $urlparams  An array of safe url parameters and their variable types, for valid values see {@link InputFilter::clean()}.
	 *
	 * @return Controller A RADController object to support chaining.
	 */
	public function display($cachable = false, array $urlparams = [])
	{
		$document = $this->app->getDocument();

		if ($document instanceof Document)
		{
			$viewType = $document->getType();
		}
		else
		{
			$viewType = $this->input->getCmd('format', 'html');
		}

		$viewName   = $this->input->get('view', $this->container->defaultView);
		$viewLayout = $this->input->get('layout', 'default');

		// Create view object
		$view = $this->getView($viewName, $viewType, $viewLayout);

		// If view has model, create the model, and assign it to the view
		if ($view->hasModel)
		{
			$model = $this->getModel($viewName);

			$view->setModel($model);
		}

		// Render the view
		$view->display();

		return $this;
	}

	/**
	 * Method to get a model object, loading it if required.
	 *
	 * @param   string  $name    The model name. Optional. Default will be the controller name
	 *
	 * @param   array   $config  Configuration array for model. Optional.
	 *
	 * @return Model The model.
	 */
	public function getModel($name = '', array $config = [])
	{
		// If name is not given, the model will has same name with controller
		if (empty($name))
		{
			$name = $this->name;
		}

		$model = $this->container->factory->createModel($name, $config);

		// Populate model state from request data
		$model->populateState();

		return $model;
	}

	/**
	 * Method to get instance of a view
	 *
	 * @param   string  $name    The view name
	 * @param   string  $type    The view type
	 * @param   string  $layout  The view layout
	 *
	 * @param   array   $config  Configuration array for view. Optional.
	 *
	 * @return HtmlView Reference to the view
	 */
	public function getView($name, $type = 'html', $layout = 'default', array $config = [])
	{
		// Merge config array with default config parameters
		$config['layout'] = $layout;
		$config['input']  = $this->input;

		if (!is_dir(JPATH_BASE . '/components/' . $this->container->option . '/View/' . ucfirst($name)))
		{
			throw new Exception(Text::sprintf('View %s not found', $name), 404);
		}

		if ($this->app->isClient('administrator'))
		{
			$config['is_admin_view'] = true;
		}

		// Set the default paths for finding the layout if it is not specified in the $config array
		if (empty($config['paths']))
		{
			$paths           = [];
			$paths[]         = JPATH_THEMES . '/' . $this->app->getTemplate()
				. '/html/' . $this->container->option . '/' . ucfirst($name);
			$paths[]         = JPATH_BASE . '/components/' . $this->container->option . '/View/'
				. ucfirst($name) . '/tmpl';
			$config['paths'] = $paths;
		}

		return $this->container->factory->createView($name, $type, $config);
	}

	/**
	 * Sets the internal message that is passed with a redirect
	 *
	 * @param   string  $text  Message to display on redirect.
	 *
	 * @param   string  $type  Message type. Optional, defaults to 'message'.
	 *
	 * @return string Previous message
	 */
	public function setMessage($text, $type = 'message')
	{
		$previous          = $this->message;
		$this->message     = $text;
		$this->messageType = $type;

		return $previous;
	}

	/**
	 * Set a URL for browser redirection.
	 *
	 * @param   string  $url   URL to redirect to.
	 *
	 * @param   string  $msg   Message to display on redirect. Optional, defaults to value set internally by controller, if any.
	 *
	 * @param   string  $type  Message type. Optional, defaults to 'message' or the type set by a previous call to setMessage.
	 *
	 * @return Controller This object to support chaining.
	 */
	public function setRedirect($url, $msg = null, $type = null)
	{
		$this->redirect = $url;

		if ($msg !== null)
		{
			// Controller may have set this directly
			$this->message = $msg;
		}

		// Ensure the type is not overwritten by a previous call to setMessage.
		if (empty($type))
		{
			if (empty($this->messageType))
			{
				$this->messageType = 'message';
			}
		}
		// If the type is explicitly set, set it.
		else
		{
			$this->messageType = $type;
		}

		return $this;
	}

	/**
	 * Redirects the browser or returns false if no redirect is set.
	 *
	 * @return boolean False if no redirect exists.
	 */
	public function redirect()
	{
		if ($this->redirect)
		{
			/* @var CMSApplication $app */
			$app = $this->app;
			$app->enqueueMessage($this->message, $this->messageType);
			$app->redirect($this->redirect);
		}

		return false;
	}

	/**
	 * Register (map) a task to a method in the class.
	 *
	 * @param   string  $task    The task name
	 *
	 * @param   string  $method  The name of the method in the derived class to perform for this task.
	 *
	 * @return Controller A Controller object to support chaining.
	 */
	public function registerTask($task, $method)
	{
		if (in_array(strtolower($method), $this->methods))
		{
			$this->taskMap[strtolower($task)] = $method;
		}

		return $this;
	}

	/**
	 * Get the application object.
	 *
	 * @return CMSApplication The application object.
	 */
	public function getApplication()
	{
		return $this->app;
	}

	/**
	 * Get the input object.
	 *
	 * @return Input The input object.
	 */
	public function getInput()
	{
		return $this->container->get('input');
	}

	/**
	 * Set controller input
	 *
	 * @param   Input  $input
	 *
	 * @return Input
	 */
	public function setInput(Input $input)
	{
		$currentInput = $this->input;

		$this->input = $input;
		$this->container->set('input', $input);
		$this->task = $input->getCmd('task', 'display');

		return $currentInput;
	}

	/**
	 * Get the last task that is being performed or was most recently performed.
	 *
	 * @return string The task that is being performed or was most recently performed.
	 */
	public function getTask()
	{
		return $this->task;
	}

	/**
	 * Set the task for controller
	 *
	 * @param $task
	 */
	public function setTask($task)
	{
		$this->task = $task;
	}

	/**
	 * Magic get method. Handles magic properties:
	 * $this->app  mapped to $this->container->app
	 * $this->input  mapped to $this->container->input
	 *
	 * @param   string  $name  The property to fetch
	 *
	 * @return  mixed|null
	 */
	public function __get($name)
	{
		$magicProperties = [
			'app',
			'option',
		];

		if (in_array($name, $magicProperties))
		{
			return $this->container->get($name);
		}

		// Property not found; raise error
		$trace = debug_backtrace();
		trigger_error(
			'Undefined property via __get(): ' . $name .
			' in ' . $trace[0]['file'] .
			' on line ' . $trace[0]['line'],
			E_USER_NOTICE
		);

		return null;
	}

	/**
	 * Check token to prevent CSRF attack
	 */
	protected function csrfProtection($method = 'post')
	{
		Session::checkToken($method) or die(Text::_('JINVALID_TOKEN'));
	}
}