<?php
/**
 * @package     OSL
 * @subpackage  Model
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\Model;

use Joomla\CMS\Filter\InputFilter;
use stdClass;

class State
{

	/**
	 * The state data container
	 *
	 * @var array
	 */
	protected $data = [];

	/**
	 * Set data for a state
	 *
	 * @param   string  $name  The name of state
	 * @param   mixed   $value
	 *
	 * @return $this
	 */
	public function set($name, $value)
	{
		if (isset($this->data[$name]))
		{
			$this->data[$name]->value = $value;
		}

		return $this;
	}

	/**
	 * Retrieve data for a state
	 *
	 * @param   string  $name     Name of the state
	 *
	 * @param   mixed   $default  Default value if no data has been set for that state
	 *
	 * @return mixed The state value
	 */
	public function get($name, $default = null)
	{
		$result = $default;

		if (isset($this->data[$name]))
		{
			$result = $this->data[$name]->value;
		}

		return $result;
	}

	/**
	 * Insert a new state
	 *
	 * @param   string  $name     The name of the state
	 *
	 * @param   mixed   $filter   The name of filter which will be used to sanitize the state value using JFilterInput
	 *
	 * @param   mixed   $default  The default value of the state
	 *
	 * @return State
	 */
	public function insert($name, $filter, $default = null)
	{
		$state             = new stdClass();
		$state->name       = $name;
		$state->filter     = $filter;
		$state->value      = $default;
		$state->default    = $default;
		$this->data[$name] = $state;

		return $this;
	}

	/**
	 * Remove an existing state
	 *
	 * @param   string  $name  The name of the state which will be removed
	 *
	 * @return $this
	 */
	public function remove($name)
	{
		if (isset($this->data[$name]))
		{
			unset($this->data[$name]);
		}

		return $this;
	}

	/**
	 * Reset all state data and revert to the default state
	 *
	 * @param   boolean  $default  If TRUE use defaults when resetting. If FALSE then null value will be used.Default is TRUE
	 *
	 * @return $this
	 */
	public function reset($default = true)
	{
		foreach ($this->data as $state)
		{
			$state->value = $default ? $state->default : null;
		}

		return $this;
	}

	/**
	 * Set the state data
	 *
	 * This function will only filter values if we have a value. If the value
	 * is an empty string it will be filtered to NULL.
	 *
	 * @param   array  $data  An associative array of state values by name
	 *
	 * @return $this
	 */
	public function setData(array $data)
	{
		$filterInput = InputFilter::getInstance();

		// Special code for handle ajax ordering in Joomla 3
		if (!empty($data['filter_full_ordering']))
		{
			$parts                    = explode(' ', $data['filter_full_ordering']);
			$sort                     = $parts[0];
			$direction                = $parts[1] ?? '';
			$data['filter_order']     = $sort;
			$data['filter_order_Dir'] = $direction;
		}

		// Filter data
		foreach ($data as $key => $value)
		{
			if (isset($this->data[$key]))
			{
				$filter = (string) $this->data[$key]->filter;

				// Only filter if we have a value
				if ($value !== null)
				{
					if ($value !== '')
					{
						// Check for a callback filter.
						if (strpos($filter, '::') !== false && is_callable(explode('::', $filter)))
						{
							$value = call_user_func(explode('::', $filter), $value);
						}

						// Filter using a callback function if specified.
						elseif (function_exists($filter))
						{
							$value = call_user_func($filter, $value);
						}

						// Filter using JFilterInput. All HTML code is filtered by default.
						else
						{
							$value = $filterInput->clean($value, $filter);
						}
					}
					else
					{
						$value = null;
					}

					$this->data[$key]->value = $value;
				}
			}
		}

		return $this;
	}

	/**
	 * Get the state data
	 *
	 * This function only returns states that have been been set.
	 *
	 * @return array An associative array of state values by name
	 */
	public function getData()
	{
		$data = [];

		foreach ($this->data as $name => $state)
		{
			$data[$name] = $state->value;
		}

		return $data;
	}

	/**
	 * Get list of state variables is being stored
	 */
	public function getProperties()
	{
		return array_keys($this->data);
	}

	/**
	 * Get default value of a state
	 *
	 * @param   string  $name
	 *
	 * @return mixed the default state value
	 */
	public function getDefault($name)
	{
		return $this->data[$name]->default;
	}

	/**
	 * Change default value (and therefore value) of an existing state
	 *
	 * @param $name
	 * @param $default
	 *
	 * @return $this
	 */
	public function setDefault($name, $default)
	{
		if (isset($this->data[$name]))
		{
			$this->data[$name]->default = $default;
			$this->data[$name]->value   = $default;
		}

		return $this;
	}

	/**
	 * Magic method to get state value
	 *
	 * @param   string
	 *
	 * @return mixed
	 */
	public function __get($name)
	{
		return $this->get($name);
	}

	/**
	 * Set state value
	 *
	 * @param   string  $name   The user-specified state name.
	 *
	 * @param   mixed   $value  The user-specified state value.
	 *
	 * @return void
	 */
	public function __set($name, $value)
	{
		$this->set($name, $value);
	}

	/**
	 * Test existence of a state variable
	 *
	 * @param   string  $name  The state name
	 *
	 * @return boolean
	 */
	public function __isset($name)
	{
		return isset($this->data[$name]);
	}

	/**
	 * Unset a state value
	 *
	 * @param   string  $name  The state name.
	 *
	 * @return void
	 */
	public function __unset($name)
	{
		if (isset($this->data[$name]))
		{
			$this->data[$name]->value = $this->data[$name]->default;
		}
	}
}