<?php
/**
 * @package        Joomla
 * @subpackage     Helpdesk Pro
 * @author         Tuan Pham Ngoc
 * @copyright      Copyright (C) 2013 - 2026 Ossolution Team
 * @license        GNU/GPL, see LICENSE.php
 */

namespace OSSolution\HelpdeskPro\Plugin\System\ReplyViaEmail\Extension;

use Ddeboer\Imap\Search\Date\Since;
use Ddeboer\Imap\Search\Flag\Unanswered;
use Ddeboer\Imap\SearchExpression;
use Ddeboer\Imap\Server;
use EmailReplyParser\Parser\EmailParser;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Cache\Cache;
use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\User\User;
use Joomla\CMS\User\UserFactoryInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\DatabaseInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use Joomla\Filesystem\File;
use OSL\Container\Container;
use OSSolution\HelpdeskPro\Admin\Model\Ticket;
use OSSolution\HelpdeskPro\Site\Helper\Helper as HelpdeskProHelper;

defined('_JEXEC') or die;

class ReplyViaEmail extends CMSPlugin implements SubscriberInterface
{
	use DatabaseAwareTrait;

	public function __construct(
		DispatcherInterface $dispatcher,
		array $config,
		CMSApplicationInterface $application,
		DatabaseInterface $db
	) {
		parent::__construct($dispatcher, $config);

		$this->setDatabase($db);
		$this->setApplication($application);
	}

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return array
	 *
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onAfterRoute' => 'onAfterRoute',
		];
	}

	/**
	 * Fetch email from mailbox and create tickets and response to tickets
	 *
	 * @param   Event  $event
	 *
	 * @return void
	 * @throws \Exception
	 */
	public function onAfterRoute(Event $event): void
	{
		if (!$this->canRun())
		{
			return;
		}

		//Store last run time
		if (!$this->params->get('secret_code', ''))
		{
			$db = $this->getDatabase();
			$this->params->set('last_run', time());

			$query = $db->getQuery(true)
				->update('#__extensions')
				->set('params = ' . $db->quote($this->params->toString()))
				->where($db->quoteName('element') . '=' . $db->quote('hdpreplyviaemail'))
				->where($db->quoteName('folder') . '=' . $db->quote('system'));

			try
			{
				// Lock the tables to prevent multiple plugin executions causing a race condition
				$db->lockTable('#__extensions');
			}
			catch (\Exception $e)
			{
				// If we can't lock the tables it's too risk continuing execution
				return;
			}

			try
			{
				// Update the plugin parameters
				$result = $db->setQuery($query)->execute();
				$this->clearCacheGroups(['com_plugins'], [0, 1]);
			}
			catch (\Exception $exc)
			{
				// If we failed to execute
				$db->unlockTables();
				$result = false;
			}

			try
			{
				// Unlock the tables after writing
				$db->unlockTables();
			}
			catch (\Exception $e)
			{
				// If we can't lock the tables assume we have somehow failed
				$result = false;
			}

			// Abort on failure
			if (!$result)
			{
				return;
			}
		}

		$this->fetNewEmails();

		$this->getApplication()->close();
	}

	/**
	 * Fetch emails and create ticket
	 *
	 * @return void
	 * @throws \Exception
	 */
	private function fetNewEmails(): void
	{
		require_once JPATH_ROOT . '/plugins/system/hdpreplyviaemail/lib/vendor/autoload.php';

		$host     = $this->params->get('host', '');
		$port     = $this->params->get('port', 993);
		$username = $this->params->get('username', '');
		$password = $this->params->get('password', '');

		$requiredParams = [$host, $username, $password];

		// Make sure all required parameters are provided before processing it further
		foreach ($requiredParams as $param)
		{
			if (!strlen(trim($param)))
			{
				return;
			}
		}

		// Attempt to connect to the mailbox
		try
		{
			// $server = new Server('mail.joomdonation.com', 993, '/imap/ssl/novalidate-cert');
			$server = new Server($host, $port, $this->buildFlags());
			// $connection is instance of \Ddeboer\Imap\Connection
			// $connection = $server->authenticate('email@domain.com', 'password');
			$connection = $server->authenticate($username, $password);
		}
		catch (\Exception $e)
		{
			$this->logData(['connect error' => $e->getMessage()]);

			// Log the error here
			return;
		}

		$mailbox = $connection->getMailbox('INBOX');

		// Search for emails from yesterday only

		$today     = new \DateTimeImmutable();
		$yesterday = $today->sub(new \DateInterval('P10D'));

		$date = Factory::getDate('Now', $this->getApplication()->get('offset'));
		$date->modify('-1 day');

		$search = new SearchExpression();
		$search->addCondition(new Unanswered());
		$search->addCondition(new Since($yesterday));

		$messages = $mailbox->getMessages($search, \SORTDATE);

		// Bootstrap the component
		require_once JPATH_ADMINISTRATOR . '/components/com_helpdeskpro/init.php';

		// Get component config data
		$config = require JPATH_ADMINISTRATOR . '/components/com_helpdeskpro/config.php';

		// Creating component container
		$container = Container::getInstance('com_helpdeskpro', $config);

		$config = HelpdeskProHelper::getConfig();

		$allowedFileTypes = explode('|', $config->allowed_file_types);

		for ($i = 0, $n = count($allowedFileTypes); $i < $n; $i++)
		{
			$allowedFileTypes[$i] = trim(strtoupper($allowedFileTypes[$i]));
		}

		$ticketIdRegex = '/#(\d+)/';

		/** @var Ticket $model */
		$model = $container->factory->createModel('Ticket', [], 'admin');

		foreach ($messages as $message)
		{
			$subject   = $message->getSubject();
			$body      = $message->getBodyText();
			$fromName  = $message->getFrom()->getName();
			$fromEmail = $message->getFrom()->getAddress();

			if ($this->ignoreEmailSubject($subject) || $this->ignoreEmail($fromEmail))
			{
				// Mark the message as ANSWERED so that it won't be processed next time
				$message->setFlag('\\ANSWERED');

				continue;
			}

			$email = (new EmailParser())->parse($body);

			$body = $email->getVisibleText() ?: $body;

			if (preg_match($ticketIdRegex, $subject, $matches))
			{
				$ticketId = (int) $matches[1];
			}
			else
			{
				$ticketId = 0;
			}

			$attachmentsPath = JPATH_ROOT . '/media/com_helpdeskpro/attachments';
			$attachments     = [];

			foreach ($message->getAttachments() as $attachment)
			{
				$filename                        = $attachment->getFilename();
				$attachments['original_names'][] = $filename;
				$filename                        = File::makeSafe($filename);
				$fileExt                         = strtoupper(HelpdeskProHelper::getFileExt($filename));

				if (in_array($fileExt, $allowedFileTypes))
				{
					if (is_file($attachmentsPath . '/' . $filename))
					{
						$filename = File::stripExt($filename) . '_' . uniqid() . '.' . $fileExt;
					}

					file_put_contents($attachmentsPath . '/' . $filename, $attachment->getDecodedContent());

					$attachments['names'][] = $filename;
				}
			}

			$data = [];

			if ($ticketId)
			{
				// Add comment
				$data['ticket_id'] = $ticketId;
				$data['message']   = $body;

				// Try to get user id from email
				$data['user_id'] = $this->getUserIdFromEmail($fromEmail);

				if (strlen(trim(strip_tags($body))))
				{
					$model->addTicketComment($data, $attachments);
				}
			}
			elseif ($this->params->get('new_ticket_category_id'))
			{
				// Add a new ticket
				$data['name']        = $fromName;
				$data['email']       = $fromEmail;
				$data['subject']     = $subject;
				$data['message']     = $body;
				$data['category_id'] = $this->params->get('new_ticket_category_id');

				if (strlen(trim(strip_tags($body))))
				{
					$model->addNewTicket($data, $attachments);
				}
			}

			// Mark the message as ANSWERED so that it won't be processed next time
			$message->setFlag('\\ANSWERED');
		}
	}

	/**
	 * Override registerListeners to only enable the plugin if
	 *
	 * @return void
	 */
	public function registerListeners()
	{
		if (!file_exists(JPATH_ROOT . '/components/com_helpdeskpro/helpdeskpro.php'))
		{
			return;
		}

		parent::registerListeners();
	}

	/**
	 * Get ID of user base on the given email
	 *
	 * @param   string  $email
	 *
	 * @return int
	 */
	private function getUserIdFromEmail(string $email): int
	{
		$db    = $this->getDatabase();
		$query = $db->getQuery(true)
			->select('id')
			->from('#__users')
			->where('email = :email')
			->bind(':email', $email);
		$db->setQuery($query);

		return (int) $db->loadResult();
	}

	/**
	 * Build the flags used for imap connection from plugin parameters
	 *
	 * @return string
	 */
	private function buildFlags()
	{
		$encryption          = $this->params->get('encryption', 'ssl');
		$validateCertificate = $this->params->get('validate_certificate', 1);

		$flags = ['imap'];

		if ($encryption)
		{
			$flags[] = $encryption;
		}

		if ($validateCertificate)
		{
			$flags[] = 'validate-cert';
		}
		else
		{
			$flags[] = 'novalidate-cert';
		}

		return '/' . implode('/', $flags);
	}

	/**
	 * Should we ignore this email subject
	 *
	 * @param   string  $subject
	 *
	 * @return bool
	 */
	private function ignoreEmailSubject($subject): bool
	{
		$subjects = explode("\r\n", $this->params->get('ignore_subjects', ''));
		$subjects = array_map('trim', $subjects);
		$subjects = array_filter($subjects, 'strlen');

		foreach ($subjects as $ignoreSubject)
		{
			if (strpos($ignoreSubject, $subject) !== false)
			{
				return true;
			}
		}

		return false;
	}

	/**
	 * Should we ignore this email
	 *
	 * @param   string  $email
	 *
	 * @return bool
	 */
	private function ignoreEmail($email): bool
	{
		if ($this->params->get('user_groups', []))
		{
			$allowUserGroups = $this->params->get('user_groups', []);

			$db    = $this->getDatabase();
			$query = $db->getQuery(true)
				->select('id')
				->from('#__users')
				->where('email = ' . $db->quote($email));
			$db->setQuery($query);
			$userId = $db->loadResult();

			// No user found, do not allow reply/adding ticket
			if (!$userId)
			{
				return true;
			}

			/* @var User $user */
			$user = Factory::getContainer()->get(UserFactoryInterface::class)->loadUserById($userId);

			/* The associated user does not belong to allow user groups, do not allow */
			if (!count(array_intersect($user->groups, $allowUserGroups)))
			{
				return true;
			}
		}

		$emails = explode("\r\n", $this->params->get('ignore_emails', ''));
		$emails = array_map('trim', $emails);
		$emails = array_filter($emails, 'strlen');

		return in_array($email, $emails);
	}

	/**
	 * Clears cache groups. We use it to clear the plugins cache after we update the last run timestamp.
	 *
	 * @param   array  $clearGroups   The cache groups to clean
	 * @param   array  $cacheClients  The cache clients (site, admin) to clean
	 *
	 * @return  void
	 *
	 * @since   2.0.4
	 */
	private function clearCacheGroups(array $clearGroups, array $cacheClients = [0, 1])
	{
		$cachePath = $this->getApplication()->get('cache_path', JPATH_SITE . '/cache');

		foreach ($clearGroups as $group)
		{
			foreach ($cacheClients as $clientId)
			{
				try
				{
					$options = [
						'defaultgroup' => $group,
						'cachebase'    => ($clientId) ? JPATH_ADMINISTRATOR . '/cache' : $cachePath,
					];

					$cache = Cache::getInstance('callback', $options);
					$cache->clean();
				}
				catch (\Exception $e)
				{
					// Ignore it
				}
			}
		}
	}

	/**
	 * Helper method to write data to a log file, for debuging purpose
	 *
	 * @param   string  $logFile
	 * @param   array   $data
	 * @param   string  $message
	 */
	private function logData($data = [], $message = null)
	{
		$text = '[' . gmdate('m/d/Y g:i A') . '] - ';

		foreach ($data as $key => $value)
		{
			$text .= "$key=$value, ";
		}

		$text .= $message;

		$fp = fopen(__DIR__ . '/logs.txt', 'a');
		fwrite($fp, $text . "\n\n");
		fclose($fp);
	}

	/**
	 * Method to check if the plugin could be run
	 *
	 * @return bool
	 */
	private function canRun(): bool
	{
		// If trigger secret_code is set, usually from cron-job request, we will process email-queues immediately
		if (trim($this->params->get('secret_code', '')))
		{
			if ($this->params->get('secret_code') == $this->getApplication()->getInput()->getString('secret_code'))
			{
				return true;
			}

			return false;
		}

		$lastRun   = (int) $this->params->get('last_run', 0);
		$now       = time();
		$cacheTime = 1200; // Every 20 minutes

		if (($now - $lastRun) < $cacheTime)
		{
			return false;
		}

		return true;
	}
}